package com.elitesland.cbpl.bpmn.data.repo;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.bpmn.data.entity.QTaskVersionDO;
import com.elitesland.cbpl.bpmn.domain.VersionStatus;
import com.elitesland.cbpl.bpmn.data.vo.param.TaskVersionPagingParamVO;
import com.elitesland.cbpl.bpmn.data.vo.param.TaskVersionQueryParamVO;
import com.elitesland.cbpl.bpmn.data.vo.resp.TaskVersionPagingVO;
import com.elitesland.cbpl.bpmn.data.vo.resp.TaskVersionRespVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import com.querydsl.jpa.impl.JPADeleteClause;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

/**
 * @author eric.hao
 * @since 2024/05/09
 */
@Component
@RequiredArgsConstructor
public class TaskVersionRepoProc {

    private final JPAQueryFactory jpaQueryFactory;
    private static final QTaskVersionDO taskVersionDO = QTaskVersionDO.taskVersionDO;

    private final QBean<TaskVersionPagingVO> taskVersionPagingVO = Projections.bean(
            TaskVersionPagingVO.class,
            taskVersionDO.id,
            taskVersionDO.masId,
            taskVersionDO.taskCode,
            taskVersionDO.status,
            taskVersionDO.version,
            taskVersionDO.taskParam,
            taskVersionDO.taskSpel,
            taskVersionDO.remark,
            taskVersionDO.createUserId,
            taskVersionDO.creator,
            taskVersionDO.createTime,
            taskVersionDO.modifyUserId,
            taskVersionDO.updater,
            taskVersionDO.modifyTime,
            taskVersionDO.deleteFlag
    );

    private Predicate pagingWhere(TaskVersionPagingParamVO query) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(taskVersionDO.deleteFlag.eq(0));
        if (ObjectUtil.isNotNull(query.getId())) {
            predicates.add(taskVersionDO.id.eq(query.getId()));
        }
        if (StrUtil.isNotBlank(query.getTaskCode())) {
            predicates.add(taskVersionDO.taskCode.eq(query.getTaskCode()));
        }
        if (ObjectUtil.isNotNull(query.getStatus())) {
            predicates.add(taskVersionDO.status.eq(query.getStatus()));
        }
        if (ObjectUtil.isNotNull(query.getVersion())) {
            predicates.add(taskVersionDO.version.eq(query.getVersion()));
        }
        if (StrUtil.isNotBlank(query.getRemark())) {
            predicates.add(taskVersionDO.remark.eq(query.getRemark()));
        }
        return ExpressionUtils.allOf(predicates);
    }

    public long taskVersionCountBy(TaskVersionPagingParamVO query) {
        var jpaQuery = jpaQueryFactory.select(taskVersionDO.id)
                .from(taskVersionDO);
        jpaQuery.where(this.pagingWhere(query));
        return jpaQuery.fetch().size();
    }

    public List<TaskVersionPagingVO> taskVersionPageBy(TaskVersionPagingParamVO query) {
        var jpaQuery = jpaQueryFactory.select(taskVersionPagingVO)
                .from(taskVersionDO);
        query.setPaging(jpaQuery);
        query.fillOrders(jpaQuery, taskVersionDO);
        jpaQuery.where(this.pagingWhere(query));
        return jpaQuery.fetch();
    }

    private final QBean<TaskVersionRespVO> taskVersionVO = Projections.bean(
            TaskVersionRespVO.class,
            taskVersionDO.id,
            taskVersionDO.masId,
            taskVersionDO.taskCode,
            taskVersionDO.status,
            taskVersionDO.version,
            taskVersionDO.taskParam,
            taskVersionDO.taskSpel,
            taskVersionDO.remark,
            taskVersionDO.createUserId,
            taskVersionDO.creator,
            taskVersionDO.createTime,
            taskVersionDO.modifyUserId,
            taskVersionDO.updater,
            taskVersionDO.modifyTime,
            taskVersionDO.deleteFlag
    );

    private Predicate where(TaskVersionQueryParamVO query) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(taskVersionDO.deleteFlag.eq(0));
        if (ObjectUtil.isNotNull(query.getId())) {
            predicates.add(taskVersionDO.id.eq(query.getId()));
        }
        if (ObjectUtil.isNotNull(query.getMasId())) {
            predicates.add(taskVersionDO.masId.eq(query.getMasId()));
        }
        if (ObjectUtil.isNotNull(query.getStatus())) {
            predicates.add(taskVersionDO.status.eq(query.getStatus()));
        }
        if (ObjectUtil.isNotNull(query.getVersion())) {
            predicates.add(taskVersionDO.version.eq(query.getVersion()));
        }
        if (StrUtil.isNotBlank(query.getRemark())) {
            predicates.add(taskVersionDO.remark.eq(query.getRemark()));
        }
        return ExpressionUtils.allOf(predicates);
    }

    public List<TaskVersionRespVO> taskVersionByParam(TaskVersionQueryParamVO query) {
        var jpaQuery = jpaQueryFactory.select(taskVersionVO)
                .from(taskVersionDO);
        jpaQuery.where(this.where(query));
        return jpaQuery.fetch();
    }

    public long delete(Long id) {
        JPADeleteClause delete = jpaQueryFactory.delete(taskVersionDO)
                .where(taskVersionDO.id.eq(id));
        return delete.execute();
    }

    public long published(Long id) {
        JPAUpdateClause update = jpaQueryFactory.update(taskVersionDO)
                .set(taskVersionDO.status, VersionStatus.EFFECTIVE.getCode())
                .where(taskVersionDO.id.eq(id));
        return update.execute();
    }

    public long unPublished(Long id, Long masId) {
        // 其他生效中 -> 发布成功
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(taskVersionDO.id.ne(id));
        predicates.add(taskVersionDO.masId.eq(masId));
        predicates.add(taskVersionDO.status.eq(VersionStatus.EFFECTIVE.getCode()));
        JPAUpdateClause update = jpaQueryFactory.update(taskVersionDO)
                .set(taskVersionDO.status, VersionStatus.PUBLISHED.getCode())
                .where(ExpressionUtils.allOf(predicates));
        return update.execute();
    }

    public int queryMaxVersion(String taskCode) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(taskVersionDO.deleteFlag.eq(0));
        predicates.add(taskVersionDO.taskCode.eq(taskCode));
        var jpaQuery = jpaQueryFactory.select(taskVersionDO.version.max())
                .from(taskVersionDO);
        jpaQuery.where(ExpressionUtils.allOf(predicates));
        Integer maxVersion = jpaQuery.fetchOne();
        return ObjectUtil.isNull(maxVersion) ? 0 : maxVersion;
    }

    public int queryEffectiveVersion(String taskCode) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(taskVersionDO.deleteFlag.eq(0));
        predicates.add(taskVersionDO.taskCode.eq(taskCode));
        predicates.add(taskVersionDO.status.eq(VersionStatus.EFFECTIVE.getCode()));
        var jpaQuery = jpaQueryFactory.select(taskVersionDO.version.max())
                .from(taskVersionDO);
        jpaQuery.where(ExpressionUtils.allOf(predicates));
        Integer maxVersion = jpaQuery.fetchOne();
        return ObjectUtil.isNull(maxVersion) ? 0 : maxVersion;
    }
}
