package com.elitesland.cbpl.bpmn.data.service.impl;

import cn.hutool.core.util.ObjectUtil;
import com.elitesland.cbpl.bpmn.data.convert.TaskVersionConvert;
import com.elitesland.cbpl.bpmn.data.repo.TaskVersionRepo;
import com.elitesland.cbpl.bpmn.domain.VersionStatus;
import com.elitesland.cbpl.bpmn.data.entity.TaskVersionDO;
import com.elitesland.cbpl.bpmn.data.repo.TaskVersionRepoProc;
import com.elitesland.cbpl.bpmn.data.service.TaskVersionService;
import com.elitesland.cbpl.bpmn.data.service.BpmnExecutor;
import com.elitesland.cbpl.bpmn.data.vo.param.TaskVersionPagingParamVO;
import com.elitesland.cbpl.bpmn.data.vo.param.TaskVersionQueryParamVO;
import com.elitesland.cbpl.bpmn.data.vo.param.TaskVersionSaveParamVO;
import com.elitesland.cbpl.bpmn.data.vo.resp.TaskVersionDetailVO;
import com.elitesland.cbpl.bpmn.data.vo.resp.TaskVersionPagingVO;
import com.elitesland.cbpl.bpmn.data.vo.resp.TaskVersionRespVO;
import com.elitesland.cbpl.tool.core.exceptions.PhoenixException;
import com.elitesland.cbpl.tool.db.PagingVO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Optional;

/**
 * @author eric.hao
 * @since 2024/05/09
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class TaskVersionServiceImpl implements TaskVersionService {

    private final TaskVersionRepo taskVersionRepo;
    private final TaskVersionRepoProc taskVersionRepoProc;
    @Autowired(required = false)
    private BpmnExecutor bpmnExecutor;

    @Override
    public PagingVO<TaskVersionPagingVO> taskVersionPageBy(TaskVersionPagingParamVO query) {
        long count = taskVersionRepoProc.taskVersionCountBy(query);
        if (count > 0) {
            var list = taskVersionRepoProc.taskVersionPageBy(query);
            return new PagingVO<>(count, list);
        }
        return new PagingVO<>();
    }

    @Override
    public List<TaskVersionRespVO> taskVersionByParam(TaskVersionQueryParamVO query) {
        return taskVersionRepoProc.taskVersionByParam(query);
    }

    @Override
    public TaskVersionDetailVO taskVersionById(Long id) {
        Optional<TaskVersionDO> taskVersionDO = taskVersionRepo.findById(id);
        if (taskVersionDO.isEmpty()) {
            throw PhoenixException.unexpected("Not Found Data");
        }
        return TaskVersionConvert.INSTANCE.doToVO(taskVersionDO.get());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public Long save(TaskVersionSaveParamVO saveParam) {
        // 新增
        if (saveParam.isNew()) {
            TaskVersionDO taskVersionDO = TaskVersionConvert.INSTANCE.saveParamToDO(saveParam);
            // 初始未发布状态
            taskVersionDO.setStatus(VersionStatus.UN_PUBLISHED.getCode());
            // 设置最新版本号
            int maxVersion = queryMaxVersion(saveParam.getTaskCode());
            taskVersionDO.setVersion(maxVersion + 1);
            taskVersionRepo.save(taskVersionDO);
            return taskVersionDO.getId();
        }
        // 修改
        else {
            Optional<TaskVersionDO> taskVersionDO = taskVersionRepo.findById(saveParam.getId());
            if (taskVersionDO.isEmpty()) {
                throw PhoenixException.unexpected("Not Found Data");
            }
            TaskVersionDO taskVersion = taskVersionDO.get();
            if (!taskVersion.getStatus().equals(VersionStatus.UN_PUBLISHED.getCode())) {
                throw new RuntimeException("仅" + VersionStatus.UN_PUBLISHED.getDesc() + "状态可编辑");
            }
            TaskVersionConvert.INSTANCE.saveParamMergeToDO(saveParam, taskVersion);
            taskVersionRepo.save(taskVersion);
            return taskVersion.getId();
        }
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public long published(Long id) {
        Optional<TaskVersionDO> versionDO = taskVersionRepo.findById(id);
        if (versionDO.isEmpty()) {
            throw new RuntimeException("发布失败，版本参数错误");
        }
        TaskVersionDO version = versionDO.get();
        // 其他生效中 -> 发布成功
        taskVersionRepoProc.unPublished(id, version.getMasId());

        if (ObjectUtil.isNotNull(bpmnExecutor)) {
            // 删除其他生效中的流程
            TaskVersionQueryParamVO query = new TaskVersionQueryParamVO();
            query.setMasId(version.getMasId());
            query.setStatus(VersionStatus.EFFECTIVE.getCode());
            List<TaskVersionRespVO> result = taskVersionByParam(query);
            result.forEach(row -> bpmnExecutor.destroy(row.getTaskCode(), row.getVersion()));

            // 重新注册流程
            bpmnExecutor.refresh(version.getTaskCode(), version.getVersion(), version.getTaskSpel());
        }

        // 发布新版本
        return taskVersionRepoProc.published(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public long delete(Long id) {
        Optional<TaskVersionDO> versionDO = taskVersionRepo.findById(id);
        if (versionDO.isEmpty()) {
            return 0;
        }
        TaskVersionDO version = versionDO.get();
        if (!version.getStatus().equals(VersionStatus.UN_PUBLISHED.getCode())) {
            throw new RuntimeException("仅" + VersionStatus.UN_PUBLISHED.getDesc() + "状态可删除");
        }
        // 销毁流程
        if (ObjectUtil.isNotNull(bpmnExecutor)) {
            bpmnExecutor.destroy(version.getTaskCode(), version.getVersion());
        }
        // TODO 删除对应的所有节点配置
        return taskVersionRepoProc.delete(id);
    }

    @Override
    public int queryMaxVersion(String taskCode) {
        return taskVersionRepoProc.queryMaxVersion(taskCode);
    }

    @Override
    public int queryEffectiveVersion(String taskCode) {
        return taskVersionRepoProc.queryEffectiveVersion(taskCode);
    }
}
