package com.elitesland.cbpl.bpmn.data.repo;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.bpmn.data.entity.QTaskDefineDO;
import com.elitesland.cbpl.bpmn.data.entity.QTaskInstanceDO;
import com.elitesland.cbpl.bpmn.data.vo.param.TaskInstanceDeleteParamVO;
import com.elitesland.cbpl.bpmn.data.vo.param.TaskInstancePagingParamVO;
import com.elitesland.cbpl.bpmn.data.vo.param.TaskInstanceSaveParamVO;
import com.elitesland.cbpl.bpmn.data.vo.resp.TaskInstancePagingVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

/**
 * @author eric.hao
 * @since 2024/05/09
 */
@Component
@RequiredArgsConstructor
public class TaskInstanceRepoProc {

    private final JPAQueryFactory jpaQueryFactory;
    private static final QTaskDefineDO taskDefineDO = QTaskDefineDO.taskDefineDO;
    private static final QTaskInstanceDO taskInstanceDO = QTaskInstanceDO.taskInstanceDO;

    private final QBean<TaskInstancePagingVO> taskInstancePagingVO = Projections.bean(
            TaskInstancePagingVO.class,
            taskInstanceDO.id,
            taskDefineDO.id.as("masId"),
            taskDefineDO.taskName,
            taskInstanceDO.taskCode,
            taskInstanceDO.version,
            taskInstanceDO.instanceStatus,
            taskInstanceDO.currNodeId,
            taskInstanceDO.currNodeName,
            taskInstanceDO.errorMessage,
            taskInstanceDO.startTime,
            taskInstanceDO.endTime,
            taskInstanceDO.remark,
            taskInstanceDO.createUserId,
            taskInstanceDO.creator,
            taskInstanceDO.createTime,
            taskInstanceDO.modifyUserId,
            taskInstanceDO.updater,
            taskInstanceDO.modifyTime,
            taskInstanceDO.deleteFlag
    );

    private Predicate pagingWhere(TaskInstancePagingParamVO query) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(taskInstanceDO.deleteFlag.eq(0));
        if (StrUtil.isNotBlank(query.getTaskCode())) {
            predicates.add(taskInstanceDO.taskCode.eq(query.getTaskCode()));
        }
        if (ObjectUtil.isNotNull(query.getVersion())) {
            predicates.add(taskInstanceDO.version.eq(query.getVersion()));
        }
        if (StrUtil.isNotBlank(query.getInstanceStatus())) {
            predicates.add(taskInstanceDO.instanceStatus.eq(query.getInstanceStatus()));
        }
        if (StrUtil.isNotBlank(query.getCurrNodeId())) {
            predicates.add(taskInstanceDO.currNodeId.eq(query.getCurrNodeId()));
        }
        if (StrUtil.isNotBlank(query.getCurrNodeName())) {
            predicates.add(taskInstanceDO.currNodeName.eq(query.getCurrNodeName()));
        }
        if (StrUtil.isNotBlank(query.getErrorMessage())) {
            predicates.add(taskInstanceDO.errorMessage.eq(query.getErrorMessage()));
        }
        if (ObjectUtil.isNotNull(query.getStartTime())) {
            predicates.add(taskInstanceDO.startTime.eq(query.getStartTime()));
        }
        if (ObjectUtil.isNotNull(query.getEndTime())) {
            predicates.add(taskInstanceDO.endTime.eq(query.getEndTime()));
        }
        if (StrUtil.isNotBlank(query.getRemark())) {
            predicates.add(taskInstanceDO.remark.eq(query.getRemark()));
        }
        return ExpressionUtils.allOf(predicates);
    }

    public long taskInstanceCountBy(TaskInstancePagingParamVO query) {
        var jpaQuery = jpaQueryFactory.select(taskInstanceDO.id)
                .from(taskInstanceDO)
                .leftJoin(taskDefineDO).on(taskDefineDO.taskCode.eq(taskInstanceDO.taskCode));
        jpaQuery.where(this.pagingWhere(query));
        return jpaQuery.fetch().size();
    }

    public List<TaskInstancePagingVO> taskInstancePageBy(TaskInstancePagingParamVO query) {
        var jpaQuery = jpaQueryFactory.select(taskInstancePagingVO)
                .from(taskInstanceDO)
                .leftJoin(taskDefineDO).on(taskDefineDO.taskCode.eq(taskInstanceDO.taskCode));
        query.setPaging(jpaQuery);
        query.fillOrders(jpaQuery, taskInstanceDO);
        jpaQuery.where(this.pagingWhere(query));
        return jpaQuery.fetch();
    }

    public long update(TaskInstanceSaveParamVO saveParam) {
        JPAUpdateClause update = jpaQueryFactory.update(taskInstanceDO)
                .set(taskInstanceDO.endTime, saveParam.getEndTime())
                .set(taskInstanceDO.errorMessage, saveParam.getErrorMessage())
                .where(taskInstanceDO.id.eq(saveParam.getId()));
        // 实例状态
        if (StrUtil.isNotBlank(saveParam.getInstanceStatus())) {
            update.set(taskInstanceDO.instanceStatus, saveParam.getInstanceStatus());
        }
        // 更新备注
        if (StrUtil.isNotBlank(saveParam.getRemark())) {
            update.set(taskInstanceDO.remark, saveParam.getRemark());
        }
        return update.execute();
    }

    public long delete(TaskInstanceDeleteParamVO paramVO) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(taskInstanceDO.taskCode.eq(paramVO.getTaskCode()));
        predicates.add(taskInstanceDO.instanceStatus.in(paramVO.getInstanceStatus()));
        if (ObjectUtil.isNotNull(paramVO.getBeforeTime())) {
            predicates.add(taskInstanceDO.startTime.lt(paramVO.getBeforeTime()));
        }
        var delete = jpaQueryFactory.delete(taskInstanceDO);
        delete.where(ExpressionUtils.allOf(predicates));
        return delete.execute();
    }
}
