package com.elitesland.cbpl.bpmn.data.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitesland.cbpl.bpmn.data.convert.TaskDefineConvert;
import com.elitesland.cbpl.bpmn.data.entity.TaskDefineDO;
import com.elitesland.cbpl.bpmn.data.repo.TaskDefineRepo;
import com.elitesland.cbpl.bpmn.data.repo.TaskDefineRepoProc;
import com.elitesland.cbpl.bpmn.registrar.execute.BpmnExecutor;
import com.elitesland.cbpl.bpmn.data.service.TaskDefineService;
import com.elitesland.cbpl.bpmn.data.service.TaskVersionService;
import com.elitesland.cbpl.bpmn.data.vo.param.*;
import com.elitesland.cbpl.bpmn.data.vo.resp.TaskDefineDetailVO;
import com.elitesland.cbpl.bpmn.data.vo.resp.TaskDefinePagingVO;
import com.elitesland.cbpl.bpmn.data.vo.resp.TaskDefineRespVO;
import com.elitesland.cbpl.bpmn.data.vo.resp.TaskVersionRespVO;
import com.elitesland.cbpl.bpmn.util.BpmnParamUtil;
import com.elitesland.cbpl.common.constant.ActiveStatus;
import com.elitesland.cbpl.common.constant.PublishStatus;
import com.elitesland.cbpl.scheduling.data.service.ScheduleConfigService;
import com.elitesland.cbpl.scheduling.registrar.DefaultSchedulingRegistrar;
import com.elitesland.cbpl.tool.core.exceptions.PhoenixException;
import com.elitesland.cbpl.tool.db.PagingVO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Optional;

/**
 * @author eric.hao
 * @since 2024/05/09
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class TaskDefineServiceImpl implements TaskDefineService {

    private final TaskDefineRepo taskDefineRepo;
    private final TaskDefineRepoProc taskDefineRepoProc;
    private final TaskVersionService taskVersionService;
    @Autowired(required = false)
    private BpmnExecutor bpmnExecutor;

    @Override
    public PagingVO<TaskDefinePagingVO> taskDefinePageBy(TaskDefinePagingParamVO query) {
        long count = taskDefineRepoProc.taskDefineCountBy(query);
        if (count > 0) {
            var list = taskDefineRepoProc.taskDefinePageBy(query);
            return new PagingVO<>(count, list);
        }
        return new PagingVO<>();
    }

    @Override
    public List<TaskDefineRespVO> taskDefineByParam(TaskDefineQueryParamVO query) {
        return taskDefineRepoProc.taskDefineByParam(query);
    }

    @Override
    public TaskDefineDetailVO taskDefineById(Long id) {
        Optional<TaskDefineDO> taskDefineDO = taskDefineRepo.findById(id);
        if (taskDefineDO.isEmpty()) {
            throw PhoenixException.unexpected("Not Found Data");
        }
        return TaskDefineConvert.INSTANCE.doToVO(taskDefineDO.get());
    }

    @Override
    public TaskDefineDetailVO effectiveDefineById(Long id) {
        TaskDefineDetailVO defineVO = taskDefineById(id);
        TaskVersionQueryParamVO query = new TaskVersionQueryParamVO();
        query.setMasId(id);
        query.setStatus(PublishStatus.EFFECTIVE.getCode());
        List<TaskVersionRespVO> version = taskVersionService.taskVersionByParam(query);

        // 唯一生效，则注册该流程
        if (CollUtil.size(version) == 1) {
            defineVO.setVersion(version.get(0).getVersion());
            defineVO.setTaskParam(version.get(0).getTaskParam());
            defineVO.setTaskSpel(version.get(0).getTaskSpel());
        }
        // 存在多个生效中的版本
        else if (CollUtil.size(version) > 1) {
            throw PhoenixException.unexpected("存在多个生效中的版本");
        } else {
            logger.warn("[BPMN] Not Found Effective SpEL.");
        }
        return defineVO;
    }


    @Transactional(rollbackFor = Exception.class)
    @Override
    public Long save(TaskDefineSaveParamVO saveParam) {
        // 新增
        if (saveParam.isNew()) {
            TaskDefineDO taskDefineDO = TaskDefineConvert.INSTANCE.saveParamToDO(saveParam);
            // 新增默认启用
            taskDefineDO.setStatus(ActiveStatus.ACTIVE.getCode());
            taskDefineRepo.save(taskDefineDO);
            return taskDefineDO.getId();
        }
        // 修改
        else {
            Optional<TaskDefineDO> taskDefineDO = taskDefineRepo.findById(saveParam.getId());
            if (taskDefineDO.isEmpty()) {
                throw PhoenixException.unexpected("Not Found Data");
            }
            TaskDefineDO taskDefine = taskDefineDO.get();
            TaskDefineConvert.INSTANCE.saveParamMergeToDO(saveParam, taskDefine);
            taskDefineRepo.save(taskDefine);
            return taskDefine.getId();
        }
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public Long updateStatusV(Long id) {
        if (ObjectUtil.isNotNull(bpmnExecutor)) {
            TaskDefineDetailVO defineVO = effectiveDefineById(id);
            // 如果流程存在
            if (bpmnExecutor.containChain(defineVO.getTaskCode(), defineVO.getVersion())) {
                bpmnExecutor.destroy(defineVO.getTaskCode(), defineVO.getVersion());
            }
            // 如果存在有效版本
            if (defineVO.existsVersion()) {
                bpmnExecutor.register(defineVO.getTaskCode(), defineVO.getVersion(), defineVO.getTaskSpel());
            }
        }
        // 启用流程
        return taskDefineRepoProc.updateStatus(id, ActiveStatus.ACTIVE.getCode());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public Long updateStatusX(Long id) {
        if (ObjectUtil.isNotNull(bpmnExecutor)) {
            TaskDefineDetailVO defineVO = effectiveDefineById(id);
            // 如果流程存在
            if (bpmnExecutor.containChain(defineVO.getTaskCode(), defineVO.getVersion())) {
                bpmnExecutor.destroy(defineVO.getTaskCode(), defineVO.getVersion());
            }
        }
        // 禁用流程
        return taskDefineRepoProc.updateStatus(id, ActiveStatus.INACTIVE.getCode());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public long updateDeleteFlag(List<Long> ids) {
        return taskDefineRepoProc.updateDeleteFlag(ids);
    }

    @Autowired(required = false)
    private DefaultSchedulingRegistrar schedulingRegistrar;
    private final ScheduleConfigService scheduleConfigService;

    @Override
    public boolean cronPublish(TaskCronPublishVO saveParam) {
        if (ObjectUtil.isNull(schedulingRegistrar)) {
            logger.error("[BPMN] Schedule Not Initialize.");
            throw new RuntimeException("[BPMN] Schedule Not Initialize.");
        }
        var defineVO = taskDefineById(saveParam.getTaskId());
        if (schedulingRegistrar.hasTask(defineVO.getTaskCode())) {
            logger.warn("[BPMN] TaskCode({}) Registered.", defineVO.getTaskCode());
            throw new RuntimeException("[BPMN] TaskCode(" + defineVO.getTaskCode() + ") Registered.");
        }
        var saveParamVO = BpmnParamUtil.bpmnParamInit(defineVO.getTaskCode(), defineVO.getTaskName(), saveParam.getCron());
        scheduleConfigService.save(saveParamVO);
        return true;
    }
}
