package com.elitesland.cbpl.bpmn.registrar.execute;

import com.elitesland.cbpl.bpmn.data.convert.TaskInstanceConvert;
import com.elitesland.cbpl.bpmn.data.service.TaskInstanceService;
import com.elitesland.cbpl.bpmn.spi.BpmnCompleteSpi;
import com.elitesland.cbpl.bpmn.util.BpmnUtil;
import com.elitesland.cbpl.bpmn.data.vo.param.TaskInstanceSaveParamVO;
import com.elitesland.cbpl.logging.syslog.util.LogUtil;
import com.elitesland.cbpl.rosefinch.constant.InstanceStatus;
import com.elitesland.cbpl.tool.core.exceptions.ExceptionUtils;
import com.elitesland.cbpl.tool.core.util.StringUtils;
import com.lzhpo.tracer.util.TracerUtils;
import com.yomahub.liteflow.flow.LiteflowResponse;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.function.Supplier;

/**
 * @author eric.hao
 * @since 2024/05/11
 */
@Slf4j
@RequiredArgsConstructor
public class InstanceExecutor {

    @Resource
    private TaskInstanceService taskInstanceService;
    @Autowired(required = false)
    private BpmnCompleteSpi bpmnCompleteSpi;

    /**
     * 实例初始化
     *
     * @param taskCode 流程任务编码
     * @param version  版本号
     * @param runType  执行方式
     * @return 保存参数
     */
    public TaskInstanceSaveParamVO instanceLog(String taskCode, Integer version, String runType) {
        try {
            TaskInstanceSaveParamVO saveParam = new TaskInstanceSaveParamVO();
            saveParam.setTraceId(TracerUtils.getTraceId());
            saveParam.setTaskCode(taskCode);
            saveParam.setVersion(version);
            saveParam.setStartTime(LocalDateTime.now());
            saveParam.setInstanceStatus(InstanceStatus.RUNNING.getCode());
            saveParam.setRemark(runType);
            Long instanceId = taskInstanceService.save(saveParam);
            saveParam.setId(instanceId);
            return saveParam;
        } catch (Exception e) {
            logger.error("[BPMN] insert instanceLog error: {}", ExceptionUtils.formatException(e));
        }
        var saveParam = new TaskInstanceSaveParamVO();
        saveParam.setId(0L);
        return saveParam;
    }

    /**
     * 记录实例日志
     *
     * @param instance 实例
     * @param supplier 执行内容
     */
    public LiteflowResponse record(TaskInstanceSaveParamVO instance, Supplier<LiteflowResponse> supplier) {
        LiteflowResponse result = BpmnUtil.errorResponse();
        // 1. 实例执行前的回调方法
        try {
            if (bpmnCompleteSpi != null) {
                var response = TaskInstanceConvert.INSTANCE.saveParamToVO(instance);
                bpmnCompleteSpi.start(response);
            }
        } catch (Exception e) {
            logger.error("[BPMN] execute complete start error.", e);
        }

        // 2. 实例执行「目前仅支持同步执行」
        Throwable throwable = null;
        try {
            result = supplier.get();
            if (result.isSuccess()) {
                // 初始状态，按实例正常执行成功处理
                instance.setInstanceStatus(InstanceStatus.COMPLETE.getCode());
            } else {
                instance.setInstanceStatus(InstanceStatus.INTERRUPT.getCode());
                instance.setErrorMessage(StringUtils.exceed(result.getMessage()));
                LogUtil.error("[BPMN][Main] 任务中断：", result.getCause());
            }
            instance.setRemark(instance.getRemark() + "；执行步骤：" + result.getExecuteStepStr());
            LogUtil.info("[BPMN][Main] 任务节点执行步骤: " + result.getExecuteStepStr());
        } catch (Exception e) {
            throwable = e;
            instance.setInstanceStatus(InstanceStatus.INTERRUPT.getCode());
            String errorMessage = ExceptionUtils.formatException(e);
            instance.setErrorMessage(StringUtils.exceed(errorMessage));
            logger.error("[BPMN] execute error: {}", errorMessage);
            LogUtil.error("[BPMN][Main] 任务执行异常：", e);
        }

        try {
            // 3. 记录实例执行结果
            instance.setEndTime(LocalDateTime.now());
            taskInstanceService.update(instance);
        } catch (Exception e) {
            logger.error("[BPMN] update instanceLog error: {}", ExceptionUtils.formatException(e));
            LogUtil.error("[BPMN][Main] 更新实例状态失败：", e);
        }

        // 4. 实例执行完后的回调方法
        try {
            if (bpmnCompleteSpi != null) {
                var response = TaskInstanceConvert.INSTANCE.saveParamToVO(instance);
                bpmnCompleteSpi.whenComplete(response, throwable);
            }
        } catch (Exception e) {
            logger.error("[BPMN] execute complete callback error.", e);
        }
        return result;
    }
}
