package com.elitesland.cbpl.cloudt.tenant;

import cn.hutool.extra.spring.SpringUtil;
import com.elitescloud.boot.context.TenantSession;
import com.elitescloud.boot.tenant.client.support.impl.DefaultTenantProvider;
import com.elitescloud.cloudt.common.common.CloudtOptional;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import lombok.NoArgsConstructor;

/**
 * 租户切换工具类
 *
 * @author eric.hao
 * @since 2023/07/10
 */
@NoArgsConstructor
public class TenantUtil {

    private static final ThreadLocal<SysTenantDTO> HAS_USED_TENANT = new ThreadLocal<>();
    private static final ThreadLocal<Boolean> HAS_USED_DEFAULT = new ThreadLocal<>();
    private static final ThreadLocal<Boolean> HAS_NOT = new ThreadLocal<>();

    /**
     * 根据租户ID，切换当前租户Session
     *
     * @param tenantId 租户ID
     * @return 当前租户
     */
    public static SysTenantDTO setCurrentTenant(Long tenantId) {
        // 如果之前不使用租户，则先取消
        var hasNot = TenantSession.getNoTenant();
        if (hasNot) {
            TenantSession.clearNoTenant();
        }
        HAS_NOT.set(hasNot);

        // 如果之前使用默认的，则先取消使用默认
        HAS_USED_DEFAULT.set(TenantSession.getUseDefault());
        if (HAS_USED_DEFAULT.get()) {
            TenantSession.clearUseDefault();
        }

        // 如果之前设置了session的租户，则先更换
        HAS_USED_TENANT.set(TenantSession.getCurrentTenant());

        // 读取对应租户，设置切换
        DefaultTenantProvider tenantProvider = SpringUtil.getBean(DefaultTenantProvider.class);
        CloudtOptional<SysTenantDTO> tenantDTO = tenantProvider.getById(tenantId);
        TenantSession.setCurrentTenant(tenantDTO.get());
        return tenantDTO.get();
    }

    /**
     * 清除当前租户的Session
     */
    public static void resetCurrentTenant() {
        // 恢复原session租户
        if (HAS_USED_TENANT.get() == null) {
            TenantSession.clearCurrentTenant();
        } else {
            TenantSession.setCurrentTenant(HAS_USED_TENANT.get());
        }
        HAS_USED_TENANT.remove();

        // 恢复原是否使用默认
        if (HAS_USED_DEFAULT.get()) {
            TenantSession.setUseDefault();
        }
        HAS_USED_DEFAULT.remove();

        // 恢复不使用
        if (HAS_NOT.get()) {
            TenantSession.setNoTenant();
        }
        HAS_NOT.remove();
    }
}
