package com.elitesland.cbpl.cloudt.jpa.auditing;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.constant.CommonConstant;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.jpa.config.auditing.AuditCustomizer;
import com.elitescloud.boot.provider.IdFactory;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitesland.cbpl.cloudt.iam.IamUserUtil;
import com.elitesland.cbpl.tool.db.entity.BaseEntity;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;

/**
 * 自定义支持云梯审计字段逻辑
 *
 * @author eric.hao
 * @since 2024/07/30
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class CustomAuditService implements AuditCustomizer {

    private static TenantClientProvider tenantClientProvider;

    @Override
    public void onCreate(Object source) {
        if (source instanceof BaseEntity) {
            BaseEntity entity = (BaseEntity) source;
            if (entity.getId() == null) {
                entity.setId(IdFactory.generateLong());
            }

            // 创建时间
            LocalDateTime now = entity.getCreateTime() == null ? LocalDateTime.now() : entity.getCreateTime();

            // 租户id
            Long tenantId = entity.getTenantId() == null ? obtainTenantId() : entity.getTenantId();

            // 当前操作用户
            Long currentUserId = 0L;
            String currentUsername = null;
            if (IamUserUtil.isLogin()) {
                currentUserId = IamUserUtil.currentUserId();
                currentUsername = IamUserUtil.currentUserName();
            }
            if (entity.getCreateUserId() != null && CharSequenceUtil.isNotBlank(entity.getCreator())) {
                currentUserId = entity.getCreateUserId();
                currentUsername = entity.getCreator();
            }

            // 设置审计字段
            entity.setCreateUserId(currentUserId);
            entity.setCreator(currentUsername);
            entity.setCreateTime(now);
            if (entity.getModifyUserId() == null || CharSequenceUtil.isBlank(entity.getUpdater())) {
                entity.setModifyUserId(currentUserId);
                entity.setUpdater(currentUsername);
            }
            entity.setModifyTime(ObjectUtil.defaultIfNull(entity.getModifyTime(), now));
            entity.setDeleteFlag(ObjectUtil.defaultIfNull(entity.getDeleteFlag(), CommonConstant.COMMON_DELETE_NO));
            entity.setTenantId(tenantId);
        }
    }

    @Override
    public void onUpdate(Object source) {
        if (source instanceof BaseEntity) {
            BaseEntity entity = (BaseEntity) source;

            // 更新用户
            if (IamUserUtil.isLogin()) {
                entity.setModifyUserId(IamUserUtil.currentUserId());
                entity.setUpdater(IamUserUtil.currentUserName());
            }

            entity.setModifyTime(LocalDateTime.now());
            entity.setDeleteFlag(ObjectUtil.defaultIfNull(entity.getDeleteFlag(), CommonConstant.COMMON_DELETE_NO));
        }
    }

    protected static Long obtainTenantId() {
        var currentTenant = getTenantClientProvider().getSessionTenant();
        if (currentTenant != null) {
            return currentTenant.getId();
        }

        return TenantConstant.DEFAULT_TENANT_ID;
    }

    protected static TenantClientProvider getTenantClientProvider() {
        if (tenantClientProvider == null) {
            tenantClientProvider = SpringContextHolder.getBean(TenantClientProvider.class);
        }
        return tenantClientProvider;
    }
}
