package com.elitesland.cbpl.cloudt.jpa.auditing;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.constant.CommonConstant;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.context.TenantOrgContext;
import com.elitescloud.boot.jpa.config.auditing.AuditCustomizer;
import com.elitescloud.boot.provider.IdFactory;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.boot.provider.UserDetailProvider;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitesland.cbpl.tool.db.entity.BaseEntity;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;

/**
 * 自定义支持云梯审计字段逻辑
 *
 * @author eric.hao
 * @since 2024/07/30
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class CustomAuditService implements AuditCustomizer {

    private static TenantClientProvider tenantClientProvider;
    private static UserDetailProvider userProvider;

    /**
     * 兼容(v0.2.8.4) 重新获取用户上下文<br/>
     * 目前使用该版本的项目：唐宫、五芳斋<br/>
     * 后续可以重构，直接引入`CustomAuditUtil.class`
     */
    @Override
    public void onCreate(Object source) {
        if (source instanceof BaseEntity) {
            BaseEntity entity = (BaseEntity) source;
            if (entity.getId() == null) {
                entity.setId(IdFactory.generateLong());
            }

            // 创建时间
            LocalDateTime now = entity.getCreateTime() == null ? LocalDateTime.now() : entity.getCreateTime();

            // 租户id
            Long tenantId = entity.getTenantId() == null ? obtainTenantId() : entity.getTenantId();

            // 当前操作用户
            var user = currentUser();
            Long currentUserId = 0L;
            String currentUsername = null;
            if (user != null && user.getUser() != null) {
                currentUserId = user.getUserId();
                currentUsername = obtainCurrentUserName(user);
            }
            if (entity.getCreateUserId() != null && CharSequenceUtil.isNotBlank(entity.getCreator())) {
                currentUserId = entity.getCreateUserId();
                currentUsername = entity.getCreator();
            }

            // 设置审计字段
            entity.setCreateUserId(currentUserId);
            entity.setCreator(currentUsername);
            entity.setCreateTime(now);
            if (entity.getModifyUserId() == null || CharSequenceUtil.isBlank(entity.getUpdater())) {
                entity.setModifyUserId(currentUserId);
                entity.setUpdater(currentUsername);
            }
            entity.setModifyTime(ObjectUtil.defaultIfNull(entity.getModifyTime(), now));
            entity.setDeleteFlag(ObjectUtil.defaultIfNull(entity.getDeleteFlag(), CommonConstant.COMMON_DELETE_NO));
            entity.setTenantId(tenantId);
        }
    }

    @Override
    public void onUpdate(Object source) {
        if (source instanceof BaseEntity) {
            BaseEntity entity = (BaseEntity) source;

            // 更新用户
            var user = currentUser();
            if (user != null && user.getUser() != null) {
                entity.setModifyUserId(user.getUserId());
                entity.setUpdater(obtainCurrentUserName(user));
            }

            entity.setModifyTime(LocalDateTime.now());
            entity.setDeleteFlag(ObjectUtil.defaultIfNull(entity.getDeleteFlag(), CommonConstant.COMMON_DELETE_NO));
        }
    }


    protected static String obtainCurrentUserName(GeneralUserDetails currentUser) {
        if (currentUser == null) {
            currentUser = currentUser();
        }
        if (currentUser == null) {
            return null;
        }

        return currentUser.getUser().getPrettyName();
    }

    protected static Long obtainTenantId() {
        var currentTenant = getTenantClientProvider().getSessionTenant();
        if (currentTenant != null) {
            return currentTenant.getId();
        }

        return TenantConstant.DEFAULT_TENANT_ID;
    }

    protected static Long obtainTenantOrgId() {
        var useTenantOrg = TenantOrgContext.isUseTenantOrg();
        if (!ObjectUtil.defaultIfNull(useTenantOrg, true)) {
            // 不使用租户组织
            return null;
        }

        return TenantOrgContext.getTenantOrg();
    }

    protected static GeneralUserDetails currentUser() {
        if (userProvider == null) {
            userProvider = SpringContextHolder.getBean(UserDetailProvider.class);
        }
        return userProvider.currentUser();
    }

    protected static TenantClientProvider getTenantClientProvider() {
        if (tenantClientProvider == null) {
            tenantClientProvider = SpringContextHolder.getBean(TenantClientProvider.class);
        }
        return tenantClientProvider;
    }
}
