package com.elitesland.cbpl.cloudt.tenant;

import cn.hutool.extra.spring.SpringUtil;
import com.elitescloud.boot.context.TenantSession;
import com.elitescloud.boot.tenant.client.common.TenantClient;
import com.elitescloud.boot.tenant.client.support.impl.DefaultTenantProvider;
import com.elitescloud.cloudt.common.common.CloudtOptional;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.util.List;

/**
 * 租户切换工具类
 *
 * @author eric.hao
 * @since 2023/07/10
 */
@Slf4j
@NoArgsConstructor
public class TenantUtil {

    private static final ThreadLocal<SysTenantDTO> HAS_USED_TENANT = new ThreadLocal<>();
    private static final ThreadLocal<Boolean> HAS_USED_DEFAULT = new ThreadLocal<>();
    private static final ThreadLocal<Boolean> HAS_NOT = new ThreadLocal<>();

    /**
     * 根据租户ID，切换当前租户Session
     *
     * @param tenantDTO 租户
     * @return 当前租户
     * @since 0.1.25-SNAPSHOT
     */
    public static SysTenantDTO setCurrentTenant(SysTenantDTO tenantDTO) {
        // 如果之前不使用租户，则先取消
        var hasNot = TenantSession.getNoTenant();
        if (hasNot) {
            TenantSession.clearNoTenant();
        }
        HAS_NOT.set(hasNot);

        // 如果之前使用默认的，则先取消使用默认
        HAS_USED_DEFAULT.set(TenantSession.getUseDefault());
        if (HAS_USED_DEFAULT.get()) {
            TenantSession.clearUseDefault();
        }

        // 如果之前设置了session的租户，则先更换
        HAS_USED_TENANT.set(TenantSession.getCurrentTenant());

        // 读取对应租户，设置切换
        TenantSession.setCurrentTenant(tenantDTO);
        return tenantDTO;
    }

    public static SysTenantDTO setCurrentTenant(Long tenantId) {
        DefaultTenantProvider tenantProvider = SpringUtil.getBean(DefaultTenantProvider.class);
        CloudtOptional<SysTenantDTO> tenantDTO = tenantProvider.getById(tenantId);
        return setCurrentTenant(tenantDTO.get());
    }

    /**
     * @since 0.1.25-SNAPSHOT
     */
    public static SysTenantDTO setCurrentTenant(String tenantCode) {
        DefaultTenantProvider tenantProvider = SpringUtil.getBean(DefaultTenantProvider.class);
        CloudtOptional<SysTenantDTO> tenantDTO = tenantProvider.getByCode(tenantCode);
        return setCurrentTenant(tenantDTO.get());
    }

    /**
     * 清除当前租户的Session
     */
    public static void resetCurrentTenant() {
        // 恢复原session租户
        if (HAS_USED_TENANT.get() == null) {
            TenantSession.clearCurrentTenant();
        } else {
            TenantSession.setCurrentTenant(HAS_USED_TENANT.get());
        }
        HAS_USED_TENANT.remove();

        // 恢复原是否使用默认
        if (HAS_USED_DEFAULT.get()) {
            TenantSession.setUseDefault();
        }
        HAS_USED_DEFAULT.remove();

        // 恢复不使用
        if (HAS_NOT.get()) {
            TenantSession.setNoTenant();
        }
        HAS_NOT.remove();
    }

    /**
     * 租户工具类「TenantProvider」是否已加载完毕
     */
    public static boolean initializeCompleted() {
        try {
            List<SysTenantDTO> allTenants = TenantClient.getAllTenants();
            log.debug("[PHOENIX-TENANT] TenantProvider initializeCompleted: {}", allTenants);
            return true;
        } catch (NullPointerException npe) {
            log.warn("[PHOENIX-TENANT] TenantProvider not initializeCompleted.");
            return false;
        } catch (Exception e) {
            // 其他错误，代码BUG，需要开发解决
            throw new RuntimeException(e);
        }
    }
}
