package com.elitesland.cbpl.data.etl.transform.template.service.impl;

import com.elitesland.cbpl.data.etl.transform.template.convert.TransformTemplateConvert;
import com.elitesland.cbpl.data.etl.transform.template.entity.TransformTemplateDO;
import com.elitesland.cbpl.data.etl.transform.template.repo.TransformTemplateRepo;
import com.elitesland.cbpl.data.etl.transform.template.repo.TransformTemplateRepoProc;
import com.elitesland.cbpl.data.etl.transform.template.service.TransformTemplateService;
import com.elitesland.cbpl.data.etl.transform.template.vo.param.TemplatePagingParamVO;
import com.elitesland.cbpl.data.etl.transform.template.vo.param.TemplateSaveParamVO;
import com.elitesland.cbpl.data.etl.transform.template.vo.resp.TransformTemplateVO;
import com.elitesland.cbpl.data.etl.transform.template.vo.resp.TemplatePagingVO;
import com.elitesland.cbpl.tool.core.exceptions.PhoenixException;
import com.elitesland.cbpl.tool.db.PagingVO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Optional;

/**
 * @author eric.hao
 * @since 2024/05/06
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class TransformTemplateServiceImpl implements TransformTemplateService {

    private final TransformTemplateRepo templateRepo;
    private final TransformTemplateRepoProc templateRepoProc;

    @Override
    public PagingVO<TemplatePagingVO> transformTemplatePageBy(TemplatePagingParamVO query) {
        long count = templateRepoProc.transformTemplateCountBy(query);
        if (count > 0) {
            var list = templateRepoProc.transformTemplatePageBy(query);
            return new PagingVO<>(count, list);
        }
        return new PagingVO<>();
    }

    @Override
    public TransformTemplateVO transformTemplateById(Long id) {
        Optional<TransformTemplateDO> templateDO = templateRepo.findById(id);
        if (templateDO.isEmpty()) {
            throw PhoenixException.unexpected("[PHOENIX-ETL] TemplateId(" + id + ") Not Found");
        }
        return TransformTemplateConvert.INSTANCE.doToVO(templateDO.get());
    }

    @Override
    public TransformTemplateVO transformTemplateByCode(String templateCode) {
        Optional<TransformTemplateDO> templateDO = templateRepo.findByTemplateCode(templateCode);
        if (templateDO.isEmpty()) {
            throw PhoenixException.unexpected("[PHOENIX-ETL] TemplateCode(" + templateCode + ") Not Found");
        }
        return TransformTemplateConvert.INSTANCE.doToVO(templateDO.get());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public Long save(TemplateSaveParamVO saveParam) {
        // 新增
        if (saveParam.isNew()) {
            TransformTemplateDO templateDO = TransformTemplateConvert.INSTANCE.saveParamToDO(saveParam);
            // 默认开启
            templateDO.setStatus(1);
            templateRepo.save(templateDO);
            return templateDO.getId();
        }
        // 修改
        else {
            Optional<TransformTemplateDO> templateDO = templateRepo.findById(saveParam.getId());
            if (templateDO.isEmpty()) {
                throw PhoenixException.unexpected("[PHOENIX-ETL] TemplateId(" + saveParam.getId() + ") Not Found");
            }
            TransformTemplateDO template = templateDO.get();
            TransformTemplateConvert.INSTANCE.saveParamMergeToDO(saveParam, template);
            templateRepo.save(template);
            return template.getId();
        }
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void updateStatus(Long id, Integer status) {
        templateRepoProc.updateStatus(id, status);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public long delete(List<Long> ids) {
        return templateRepoProc.delete(ids);
    }
}
