package com.elitesland.cbpl.data.etl.transform.template.controller;

import cn.hutool.json.JSONConfig;
import cn.hutool.json.JSONUtil;
import com.elitesland.cbpl.data.etl.transform.template.service.TransformTemplateService;
import com.elitesland.cbpl.data.etl.transform.template.vo.param.TemplatePagingParamVO;
import com.elitesland.cbpl.data.etl.transform.template.vo.param.TemplateSaveParamVO;
import com.elitesland.cbpl.data.etl.transform.template.vo.resp.TemplatePagingVO;
import com.elitesland.cbpl.data.etl.transform.template.vo.resp.TransformTemplateVO;
import com.elitesland.cbpl.data.etl.util.DataETL;
import com.elitesland.cbpl.tool.api.domain.HttpResult;
import com.elitesland.cbpl.tool.db.PagingVO;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * @author eric.hao
 * @since 2024/05/06
 */
@Slf4j
@Api(tags = "ETL-转换模板表")
@RestController
@RequestMapping("/etl/transform")
@AllArgsConstructor
public class TransformTemplateController {

    private final TransformTemplateService transformTemplateService;

    @ApiOperation(value = "分页查询")
    @PostMapping(value = "/list")
    public HttpResult<PagingVO<TemplatePagingVO>> list(@RequestBody TemplatePagingParamVO query) {
        logger.info("[TransformTemplate] query paging param = {}", query);
        return HttpResult.ok(transformTemplateService.transformTemplatePageBy(query));
    }

    @ApiOperation(value = "明细查询")
    @GetMapping(value = "/detail/{id}")
    public HttpResult<TransformTemplateVO> detail(@PathVariable("id") Long id) {
        logger.info("[TransformTemplate] query detail by id = {}", id);
        return HttpResult.ok(transformTemplateService.transformTemplateById(id));
    }

    @ApiOperation(value = "新增或修改")
    @PostMapping(value = "/save")
    public HttpResult<Long> save(@RequestBody TemplateSaveParamVO saveParam) {
        logger.info("[TransformTemplate] create/update record, saveParam = {}", saveParam);
        return HttpResult.ok(transformTemplateService.save(saveParam));
    }

    @ApiOperation(value = "启用")
    @PatchMapping(value = "/{id}/v")
    public HttpResult<Void> active(@PathVariable("id") Long id) {
        logger.info("[TransformTemplate] active by id = {}", id);
        transformTemplateService.updateStatus(id, 1);
        return HttpResult.ok();
    }

    @ApiOperation(value = "禁用")
    @PatchMapping(value = "/{id}/x")
    public HttpResult<Void> inactive(@PathVariable("id") Long id) {
        logger.info("[TransformTemplate] inactive by id = {}", id);
        transformTemplateService.updateStatus(id, 0);
        return HttpResult.ok();
    }

    @ApiOperation(value = "批量物理删除")
    @DeleteMapping(value = "/delete")
    public HttpResult<Long> delete(@RequestBody List<Long> ids) {
        logger.info("[TransformTemplate] batch physical delete by ids = {}", ids);
        return HttpResult.ok(transformTemplateService.delete(ids));
    }

    /**
     * @since 0.1.3-SNAPSHOT
     */
    @ApiOperation(value = "测试验证数据转换结果")
    @PostMapping("/verify")
    public Object verify(String templateCode, @RequestBody Object data) {
        String result = DataETL.transform(templateCode, data);
        return JSONUtil.wrap(result, new JSONConfig());
    }
}
