package com.elitesland.cbpl.data.etl.util;

import cn.hutool.extra.spring.SpringUtil;
import com.elitesland.cbpl.data.etl.transform.template.service.TransformTemplateService;
import com.elitesland.cbpl.data.etl.transform.template.service.impl.TransformTemplateServiceImpl;
import com.elitesland.cbpl.data.etl.transform.template.vo.resp.TransformTemplateVO;
import com.elitesland.cbpl.data.etl.transform.util.TransformUtil;
import com.elitesland.cbpl.tool.core.bean.BeanUtils;
import com.elitesland.cbpl.tool.core.exceptions.PhoenixException;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.extern.slf4j.Slf4j;
import org.apache.velocity.VelocityContext;

import static com.elitesland.cbpl.data.etl.config.DataCleanProperties.DATA_ETL_ENABLED;

/**
 * @author eric.hao
 * @since 2024/05/06
 */
@Slf4j
public class DataETL {

    private static TransformTemplateService templateService;

    private static TransformTemplateService getTemplateInstance() {
        if (templateService == null) {
            templateService = SpringUtil.getBean(TransformTemplateServiceImpl.class);
        }
        return templateService;
    }

    /**
     * 根据模板，数据转换
     *
     * @param templateCode 模板编码
     * @param data         数据上下文
     * @return 转换后数据
     */
    public static String transform(String templateCode, Object data) {
        if (!DATA_ETL_ENABLED) {
            logger.warn("[PHOENIX][ETL] data-etl not enabled.");
            return BeanUtils.toJsonStr(data);
        }
        TransformTemplateVO template = getTemplateInstance().transformTemplateByCode(templateCode);
        if (!template.available()) {
            throw PhoenixException.unchecked("[PHOENIX][ETL] 模板编码(" + templateCode + ")不可用.");
        }
        // 模板上下文
        VelocityContext context = new VelocityContext();
        context.put("ctx", data);
        return TransformUtil.parseToJsonStr(context, template.getTemplateCode(), template.getContent());
    }

    /**
     * 转换到指定类
     */
    public static <T> T transform(String templateCode, Object data, Class<T> clazz) {
        String transform = transform(templateCode, data);
        return BeanUtils.toBean(transform, clazz);
    }

    /**
     * 转换到指定类
     */
    public static <T> T transform(String templateCode, Object data, TypeReference<T> typeRef) {
        String transform = transform(templateCode, data);
        return BeanUtils.toBean(transform, typeRef);
    }
}
