package com.elitesland.cbpl.infinity.client.router.service;

import cn.hutool.core.lang.Assert;
import cn.hutool.http.HttpUtil;
import com.elitesland.cbpl.infinity.client.router.builder.RouterParamBuilder;
import com.elitesland.cbpl.infinity.server.router.vo.param.InfinityRouterParamVO;
import com.elitesland.cbpl.infinity.web.common.vo.InfinityResponseVO;
import com.elitesland.cbpl.infinity.web.http.param.HttpParam;
import com.elitesland.cbpl.infinity.web.http.service.InfinityRestClient;
import com.elitesland.cbpl.tool.core.bean.BeanUtils;
import com.elitesland.cbpl.tool.core.exceptions.ExceptionUtils;
import com.elitesland.cbpl.tool.core.http.RestWrapper;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import static com.elitesland.cbpl.infinity.client.config.InfinityClientProperties.INFINITY_CLIENT_ENABLED;

/**
 * @author eric.hao
 * @since 2023/04/06
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class RouterClientServiceImpl implements RouterClientService {

    private final RouterParamBuilder routerParamBuilder;
    private final RestWrapper restWrapper;
    private final InfinityRestClient infinityRestClient;
    @Autowired
    private RestTemplate restTemplate;

    @Override
    public <T, K extends InfinityResponseVO> List<T> query(InfinityRouterParamVO param, Map<String, Object> requestBody, Class<T> resultType, Class<K> responseType) {
        Assert.isTrue(INFINITY_CLIENT_ENABLED, "[PHOENIX-INFINITY] client disabled.");
        HttpParam<T, K> routerParam = routerParamBuilder.routerParamBuilder(param, requestBody, resultType, responseType);
        try {
            K response = infinityRestClient.exchangeToBean(routerParam);
            if (InfinityResponseVO.success(response, "S")) {
                List<Map<String, Object>> entities = response.getData();
                log.trace("[INFINITY-HTTP] RESULT LIST: {}", entities);
                return BeanUtils.toList(entities, routerParam.getResultType());
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        log.error("[INFINITY-HTTP] QUERY FAIL. URL({}), PARAMS:({})", routerParam.getUrl(), BeanUtils.toJsonStr(routerParam.getBody()));
        return Collections.emptyList();
    }

    @Override
    public <K extends InfinityResponseVO> K send(InfinityRouterParamVO param, Map<String, Object> requestBody, Class<K> responseType) {
        return sendObj(param, requestBody, responseType);
    }

    @Override
    public <K extends InfinityResponseVO> K send(InfinityRouterParamVO param, List<Map<String, Object>> requestBody, Class<K> responseType) {
        return sendObj(param, requestBody, responseType);
    }
    private <K extends InfinityResponseVO> K sendObj(InfinityRouterParamVO param, Object requestBody, Class<K> responseType) {
        Assert.isTrue(INFINITY_CLIENT_ENABLED, "[PHOENIX-INFINITY] client disabled.");
        String message;
        HttpParam<?, K> routerParam = routerParamBuilder.routerParamBuilder(param, requestBody, null, responseType);
        try {
            return infinityRestClient.exchangeToBean(routerParam);
        } catch (Exception e) {
            message = ExceptionUtils.formatException(e);
            e.printStackTrace();
        }
        log.error("[INFINITY-HTTP] PUT FAIL. URL({}), PARAMS:({})", routerParam.getUrl(), BeanUtils.toJsonStr(routerParam.getBody()));
        InfinityResponseVO error = InfinityResponseVO.error(message);
        return BeanUtils.toBean(BeanUtils.toJsonStr(error), responseType);
    }

    @Override
    public String exchange(InfinityRouterParamVO param, Map<String, Object> requestBody) {
        Assert.isTrue(INFINITY_CLIENT_ENABLED, "[PHOENIX-INFINITY] client disabled.");
        var routerParam = routerParamBuilder.routerParamBuilder(param, requestBody, null, null);
        try {
            return infinityRestClient.exchange(routerParam);
        } catch (Exception e) {
            e.printStackTrace();
        }
        log.error("[INFINITY-HTTP] PUT FAIL. URL({}), PARAMS:({})", routerParam.getUrl(), BeanUtils.toJsonStr(routerParam.getBody()));
        return "";
    }

    @Override
    public Resource exchangeStream(InfinityRouterParamVO param, Map<String, Object> requestBody) {
        Assert.isTrue(INFINITY_CLIENT_ENABLED, "[PHOENIX-INFINITY] client disabled.");
        var routerParam = routerParamBuilder.routerParamBuilder(param, requestBody, null, null);
        try {
            String queryParams = HttpUtil.toParams(routerParam.getBodyMap(), StandardCharsets.UTF_8);
            String requestUrl = routerParam.getUrl() + "?" + queryParams;
            HttpEntity<String> entity = restWrapper.getMethodHttpEntity(routerParam.getHeaders());
            return restTemplate.exchange(URI.create(requestUrl), HttpMethod.GET, entity, Resource.class).getBody();
        } catch (Exception e) {
            e.printStackTrace();
        }
        log.error("[INFINITY-HTTP] PUT FAIL. URL({}), PARAMS:({})", routerParam.getUrl(), BeanUtils.toJsonStr(routerParam.getBody()));
        return null;
    }
}
