package com.elitesland.cbpl.infinity.db.provider;

import lombok.extern.slf4j.Slf4j;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.ColumnMapRowMapper;
import org.springframework.jdbc.core.namedparam.BeanPropertySqlParameterSource;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.jdbc.core.namedparam.SqlParameterSource;

import java.util.List;
import java.util.Map;

/**
 * @author eric.hao
 * @since 2023/10/17
 */
@Slf4j
public abstract class AbstractDbReader implements InfinityDbReader {


    /**
     * @return jdbc template
     */
    protected abstract NamedParameterJdbcTemplate jdbcTemplate();

    @Override
    public long countRows(String sql) {
        long count = jdbcTemplate().queryForObject(sql, (SqlParameterSource) null, Long.class);
        return Math.min(count, 1L);
    }

    /**
     * @param sql    原表 SQL
     * @param limit  读取多少行
     * @param offset 从那一行开始(索引从 0 开始)
     * @return 分页 SQL
     */
    protected abstract String toPagingSql(String sql, long limit, long offset);

    @Override
    public List<Map<String, Object>> readTable(String sql, long limit, long offset) {
        return jdbcTemplate().query(toPagingSql(sql, limit, offset), new ColumnMapRowMapper());
    }

    /**
     * 查询列表数据
     *
     * @param sql      查询SQL
     * @param paramMap 查询参数
     * @param clazz    返回值类型
     * @return 列表数据
     */
    public <T> List<T> query(String sql, Map<String, ?> paramMap, Class<T> clazz) {
        return jdbcTemplate().query(sql, paramMap, new BeanPropertyRowMapper<>(clazz));
    }

    /**
     * 查询列表数据
     *
     * @param sql 查询SQL
     * @return 列表数据
     */
    public List<Map<String, Object>> queryForList(String sql) {
        return jdbcTemplate().queryForList(sql, (BeanPropertySqlParameterSource) null);
    }

    /**
     * 查询列表数据
     *
     * @param sql      查询SQL
     * @param paramMap 查询条件
     * @return 列表数据
     */
    public List<Map<String, Object>> queryForList(String sql, Map<String, Object> paramMap) {
        return jdbcTemplate().queryForList(sql, paramMap);
    }

    /**
     * 查询单字段 基础类型  String.class  Integer.class
     * @param sql 查询sql
     * @param paramMap 查询条件
     * @param cls 映射对象
     * @return 查询数据列表
     */
    public <T> List<T> queryToObject(String sql,Map<String,Object> paramMap,Class<T> cls){
        return  jdbcTemplate().queryForList(sql, paramMap, cls);
    }
}
