package com.elitesland.cbpl.infinity.db.provider;

import com.elitesland.cbpl.infinity.db.util.JdbcUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.jdbc.core.namedparam.SqlParameterSource;

import java.util.List;
import java.util.Map;

/**
 * @author eric.hao
 * @since 2023/10/17
 */
@Slf4j
public abstract class AbstractDbWriter implements InfinityDbWriter {

    /**
     * @return jdbc template type (eg. mysql, oracle, etc.)
     */
    protected abstract String jdbcTemplateType();

    /**
     * @return jdbc template
     */
    protected abstract NamedParameterJdbcTemplate jdbcTemplate();

    @Override
    @SuppressWarnings("unchecked")
    public long writeTable(String tableName, List<Map<String, Object>> rows, boolean append) {
        if (!append) {
            jdbcTemplate().update(buildClearSql(tableName), (SqlParameterSource) null);
            log.info("[INFINITY-DB][AbstractDbWriter] 1. {} ({}) 已清除.", tableName, jdbcTemplateType());
        }
        if (rows.isEmpty()) {
            log.info("[INFINITY-DB][AbstractDbWriter] 2. 同步源没有数据.");
            return 0;
        }
        String sql = JdbcUtil.buildInsertSql(tableName, rows.get(0).keySet());
        int[] results = jdbcTemplate().batchUpdate(sql, rows.toArray(new Map[0]));
        log.info("[INFINITY-DB][AbstractDbWriter] 3. {} ({}) 已清除.", tableName, results);
        return JdbcUtil.countResults(results);
    }

    public String buildClearSql(String tableName) {
        return "delete from " + tableName;
    }

    public void deleteTableById(String tableName, String sqlStr) {
        String s = buildClearSql(tableName + " where " + sqlStr);
        final int update = jdbcTemplate().update(buildClearSql(tableName + " where " + sqlStr), (SqlParameterSource) null);
        log.info("[INFINITY-DB][AbstractDbWriter] {} ({}) 已清除.", tableName, update);
    }

    public int update(String sql, Map<String, Object> line) {
        return jdbcTemplate().update(sql, new MapSqlParameterSource(line));
    }
}
