package com.elitesland.cbpl.infinity.db.source.service;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitesland.cbpl.infinity.db.source.loader.DataSourceLoader;
import com.elitesland.cbpl.infinity.db.source.loader.DefaultSettingLoader;
import com.elitesland.cbpl.infinity.db.source.provider.DataSourceProvider;
import com.elitesland.cbpl.infinity.db.source.vo.DataSourceVO;
import com.zaxxer.hikari.HikariDataSource;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;

import javax.annotation.PostConstruct;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

/**
 * @author eric.hao
 * @since 2023/12/19
 */
@Slf4j
@RequiredArgsConstructor
public class DataSourceService {

    private final DataSourceLoader dataSourceLoader;

    private final Map<String, DataSourceProvider> providers = new HashMap<>();

    @PostConstruct
    public void init() {
        List<DataSourceVO> ds = dataSourceLoader.loadDataSource();
        if (CollUtil.isEmpty(ds)) {
            log.info("[PHOENIX-INFINITY] load datasource empty.");
            return;
        }

        ds.forEach(this::createDataSource);
    }

    /**
     * 初始化数据源
     */
    private void createDataSource(DataSourceVO ds) {
        Properties properties = ObjectUtil.isEmpty(ds.getProperties()) ? DefaultSettingLoader.getProperties(ds.getName()) : ds.getProperties();

        var dataSource = new HikariDataSource();
        dataSource.setJdbcUrl(ds.getUrl());
        dataSource.setUsername(ds.getUsername());
        dataSource.setPassword(ds.getPassword());
        dataSource.setDriverClassName(ds.getDriverClassName());
        dataSource.setDataSourceProperties(properties);

        providers.put(ds.getName(), DataSourceProvider.setDatasource(dataSource));
    }

    /**
     * 读取指定名称的数据源
     *
     * @param name 数据源名称
     * @return jdbcTemplate
     */
    public NamedParameterJdbcTemplate getJdbcTemplate(String name) {
        DataSourceProvider provider = providers.get(name);
        if (ObjectUtil.isNull(provider)) {
            throw new RuntimeException("读取失败，未找到数据源");
        }
        if (!provider.isInitialized()) {
            throw new RuntimeException("读取失败，数据源未初始化");
        }
        return provider.getJdbcTemplate();
    }
}
