package com.elitesland.cbpl.infinity.web.http.service;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.infinity.server.api.domain.FormColumnDefinition;
import com.elitesland.cbpl.infinity.server.api.service.InfinityApiService;
import com.elitesland.cbpl.infinity.web.http.domain.InfinityResult;
import com.elitesland.cbpl.infinity.web.http.param.HttpParam;
import com.elitesland.cbpl.infinity.web.security.service.SecurityAuthService;
import com.elitesland.cbpl.unicom.util.UnicomClient;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.MediaType;
import org.springframework.util.MultiValueMap;
import org.springframework.web.reactive.function.client.WebClient;
import org.springframework.web.util.UriComponentsBuilder;

import javax.annotation.Resource;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.function.Function;

/**
 * @author eric.hao
 * @since 2024/05/20
 */
@Slf4j
@RequiredArgsConstructor
public class InfinityRestClient {

    private final WebClient webClient;
    @Resource
    private InfinityApiService infinityApiService;
    @Resource
    private SecurityAuthService securityAuthService;

    /**
     * 无参数，返回字符传
     */
    public InfinityResult<String> exchange(String apiCode) {
        return exchange(apiCode, String.class, null);
    }

    /**
     * 无参数，返回指定类型
     */
    public <R> InfinityResult<R> exchange(String apiCode, Class<R> responseType) {
        return exchange(apiCode, responseType, null);
    }

    /**
     * 自定义参数，返回字符传
     */
    public InfinityResult<String> exchange(String apiCode, Function<HttpParam.HttpParamBuilder, HttpParam> func) {
        return exchange(apiCode, String.class, func);
    }

    /**
     * 自定义参数，返回指定类型
     */
    public <R> InfinityResult<R> exchange(String apiCode, Class<R> responseType, Function<HttpParam.HttpParamBuilder, HttpParam> fn) {
        var builder = HttpParam.builder().apiConfig(infinityApiService.infinityApiByApiCode(apiCode));
        HttpParam param = fn != null ? fn.apply(builder) : builder.build();
        // 1. 接口地址
        UriComponentsBuilder uriBuilder = UriComponentsBuilder.fromHttpUrl(param.getUrl());
        // ====>>>>> TODO 配置项：是否强制按照接口规范，进行参数转换
        // 2.1 queryParam固定参数
        List<FormColumnDefinition> queryConfig = param.getApiConfig().getQueryParam();
        if (CollUtil.isNotEmpty(queryConfig)) {
            queryConfig.stream()
                    .filter(def -> StrUtil.isNotBlank(def.getFixedValue()))
                    .forEach(def -> uriBuilder.queryParam(def.getColumnKey(), def.getFixedValue()));
        }
        // 2.2 queryParam动态参数
        if (MapUtil.isNotEmpty(param.getQueryParam())) {
            uriBuilder.queryParams(param.getQueryParam());
        }
        // 3. 请求方式、pathParam
        WebClient.RequestBodyUriSpec method = webClient.method(param.getRequestMethod());
        WebClient.RequestBodySpec uri = MapUtil.isEmpty(param.getPathParam()) ? method.uri(uriBuilder.toUriString())
                : method.uri(uriBuilder.toUriString(), param.getPathParam());
        uri.acceptCharset(StandardCharsets.UTF_8);

        // TODO 4. 认证方式(authMethod): 如何取值，如何认证
        UnicomClient.supply(() -> securityAuthService.authentication(uri, param), param.getAuthMethod());

        // 5. headerParam
        uri.headers(headers -> {
            // 5.1 固定参数
            List<FormColumnDefinition> headerConfig = param.getApiConfig().getHeaderParam();
            if (CollUtil.isNotEmpty(headerConfig)) {
                headerConfig.stream()
                        .filter(def -> StrUtil.isNotBlank(def.getFixedValue()))
                        .forEach(def -> headers.set(def.getColumnKey(), def.getFixedValue()));
            }
            // 5.2 动态参数
            MultiValueMap<String, String> headerParam = param.getHeaderParam();
            if (MapUtil.isNotEmpty(headerParam)) {
                for (String key : headerParam.keySet()) {
                    headers.set(key, headerParam.getFirst(key));
                }
            }
        });
        // 6. 根据`mediaType`, 读取`bodyParam`进行提交
        if (param.needBodyParam()) {
            uri.contentType(MediaType.parseMediaType(param.getMediaType()));
            if (param.formData()) {
                uri.body(param.getBodyFormData());
            } else if (param.objectData()) {
                uri.bodyValue(param.getBodyParam());
            }
        }
        return InfinityResult.of(responseType, param, uri);
    }
}
