package com.elitesland.cbpl.infinity.server.api.vo.resp;

import com.elitesland.cbpl.common.constant.ActiveStatus;
import com.elitesland.cbpl.common.constant.AuthMethod;
import com.elitesland.cbpl.infinity.web.security.domain.BasicAuthAccountVO;
import com.elitesland.cbpl.infinity.web.security.domain.OAuth2AccountVO;
import com.elitesland.cbpl.infinity.web.security.util.AuthAccountResolve;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import lombok.Data;
import lombok.NonNull;
import org.springframework.http.HttpMethod;

import javax.persistence.EnumType;
import javax.persistence.Enumerated;
import java.time.LocalDateTime;
import java.util.List;

/**
 * 接口定义
 *
 * @author eric.hao
 * @since 2024/05/16
 */
@Data
@ApiModel("接口定义")
@JsonInclude(JsonInclude.Include.NON_NULL)
public class InfinityApiRespVO implements Comparable<InfinityApiRespVO> {

    @ApiModelProperty(value = "唯一标识")
    private Long id;

    @ApiModelProperty(value = "平台编码")
    private String platformCode;

    @ApiModelProperty(value = "分组id")
    private Long folderId;

    @ApiModelProperty(value = "目录名称")
    private String folderName;

    @ApiModelProperty(value = "服务url前缀")
    private String serverUrl;

    @ApiModelProperty(value = "认证方式")
    private String authMethod;

    @ApiModelProperty(value = "认证方式中文")
    private String authMethodName;

    public String getAuthMethodName() {
        return AuthMethod.fromCode(authMethod);
    }

    @JsonIgnore
    @ApiModelProperty(value = "认证账号")
    private String authAccount;

    @ApiModelProperty(value = "BasicAuth 认证账号")
    private BasicAuthAccountVO basicAuthAccount;

    public BasicAuthAccountVO getBasicAuthAccount() {
        return AuthAccountResolve.getBasicAuthAccount(authMethod, authAccount);
    }

    @ApiModelProperty(value = "OAuth2.0 认证账号")
    private OAuth2AccountVO oauth2Account;

    public OAuth2AccountVO getOauth2Account() {
        return AuthAccountResolve.getOauth2Account(authMethod, authAccount);
    }

    @ApiModelProperty(value = "接口编码")
    private String apiCode;

    @ApiModelProperty(value = "接口名称")
    private String apiName;

    @ApiModelProperty(value = "接口地址")
    private String apiUrl;

    @ApiModelProperty(value = "请求方式")
    @Enumerated(EnumType.STRING)
    private HttpMethod requestMethod;

    @ApiModelProperty(value = "接口状态")
    private Integer apiStatus;

    @ApiModelProperty(value = "接口状态中文")
    private String apiStatusName;

    public String getApiStatusName() {
        return ActiveStatus.fromCode(apiStatus);
    }

    @ApiModelProperty(value = "备注")
    private String remark;

    @ApiModelProperty(value = "记录创建者id")
    private Long createUserId;

    @ApiModelProperty(value = "记录创建者")
    private String creator;

    @ApiModelProperty(value = "记录创建时间")
    private LocalDateTime createTime;

    @ApiModelProperty(value = "记录最后更新者id")
    private Long modifyUserId;

    @ApiModelProperty(value = "记录最后更新者")
    private String updater;

    @ApiModelProperty(value = "记录最后更新时间")
    private LocalDateTime modifyTime;

    @ApiModelProperty(value = "子节点")
    private List<InfinityApiRespVO> children;

    public InfinityApiRespVO folderNode() {
        InfinityApiRespVO result = new InfinityApiRespVO();
        result.setFolderId(this.getFolderId());
        result.setFolderName(this.getFolderName());
        result.setServerUrl(this.getServerUrl());
        result.setAuthMethod(this.getAuthMethod());
        result.setAuthMethodName(this.getAuthMethodName());
        return result;
    }

    /**
     * 按中文名称排序（目录 or 接口）
     */
    @Override
    public int compareTo(@NonNull InfinityApiRespVO prev) {
        // 有目录的排前面，没有目录的排后面
        if (prev.getFolderId() != null && this.getFolderId() == null) {
            return 1;
        }
        if (prev.getFolderId() == null && this.getFolderId() != null) {
            return -1;
        }
        if (prev.getFolderId() != null && this.getFolderId() != null) {
            int folderSort = this.getFolderId().compareTo(prev.getFolderId());
            // 相同目录，按接口名称排序
            if (folderSort == 0) {
                return this.getApiName().compareTo(prev.getApiName());
            }
            // 不同目录，按目录名称排序
            else {
                return this.getFolderName().compareTo(prev.getFolderName());
            }
        }
        return this.getApiName().compareTo(prev.getApiName());
    }
}
