package com.elitesland.cbpl.infinity.server.account.controller;

import com.elitesland.cbpl.tool.api.domain.HttpResult;
import com.elitesland.cbpl.tool.db.PagingVO;
import com.elitesland.cbpl.infinity.server.account.service.InfinityAccountService;
import com.elitesland.cbpl.infinity.server.account.vo.param.InfinityAccountPagingParamVO;
import com.elitesland.cbpl.infinity.server.account.vo.param.InfinityAccountQueryParamVO;
import com.elitesland.cbpl.infinity.server.account.vo.param.InfinityAccountSaveParamVO;
import com.elitesland.cbpl.infinity.server.account.vo.resp.InfinityAccountDetailVO;
import com.elitesland.cbpl.infinity.server.account.vo.resp.InfinityAccountPagingVO;
import com.elitesland.cbpl.infinity.server.account.vo.resp.InfinityAccountRespVO;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * @author eric.hao
 * @since 2024/05/16
 */
@Slf4j
@Api(tags = "对外提供接口的认证账号")
@RestController
@RequestMapping("/infinity/server/account")
@AllArgsConstructor
public class InfinityAccountController {

    private final InfinityAccountService infinityAccountService;

    @ApiOperation(value = "分页查询")
    @PostMapping(value = "/list")
    public HttpResult<PagingVO<InfinityAccountPagingVO>> list(@RequestBody InfinityAccountPagingParamVO query) {
        logger.info("[InfinityAccount] query paging param = {}", query);
        return HttpResult.ok(infinityAccountService.infinityAccountPageBy(query));
    }

    @ApiOperation(value = "不分页查询")
    @PostMapping(value = "/query")
    public HttpResult<List<InfinityAccountRespVO>> query(@RequestBody InfinityAccountQueryParamVO query) {
        logger.info("[InfinityAccount] query list param = {}", query);
        return HttpResult.ok(infinityAccountService.infinityAccountByParam(query));
    }

    @ApiOperation(value = "明细查询")
    @GetMapping(value = "/detail/{id}")
    public HttpResult<InfinityAccountDetailVO> detail(@PathVariable("id") Long id) {
        logger.info("[InfinityAccount] query detail by id = {}", id);
        return HttpResult.ok(infinityAccountService.infinityAccountById(id));
    }

    @ApiOperation(value = "新增")
    @PostMapping(value = "/create")
    public HttpResult<Long> create(@RequestBody InfinityAccountSaveParamVO saveParam) {
        logger.info("[InfinityAccount] create one record, saveParam = {}", saveParam);
        return HttpResult.ok(infinityAccountService.create(saveParam));
    }

    @ApiOperation(value = "更新")
    @PutMapping(value = "/update")
    public HttpResult<Long> update(@RequestBody InfinityAccountSaveParamVO saveParam) {
        logger.info("[InfinityAccount] update one record, saveParam = {}", saveParam);
        return HttpResult.ok(infinityAccountService.update(saveParam));
    }

    @ApiOperation(value = "新增或修改")
    @PostMapping(value = "/save")
    public HttpResult<Long> save(@RequestBody InfinityAccountSaveParamVO saveParam) {
        logger.info("[InfinityAccount] create/update record, saveParam = {}", saveParam);
        return HttpResult.ok(infinityAccountService.save(saveParam));
    }

    @ApiOperation(value = "启用")
    @PatchMapping(value = "/{id}/v")
    public HttpResult<Void> active(@PathVariable("id") Long id) {
        logger.info("[InfinityAccount] active by id = {}", id);
        infinityAccountService.updateStatus(id, "1");
        return HttpResult.ok();
    }

    @ApiOperation(value = "禁用")
    @PatchMapping(value = "/{id}/x")
    public HttpResult<Void> inactive(@PathVariable("id") Long id) {
        logger.info("[InfinityAccount] inactive by id = {}", id);
        infinityAccountService.updateStatus(id, "0");
        return HttpResult.ok();
    }

    @ApiOperation(value = "批量物理删除")
    @DeleteMapping(value = "/delete")
    public HttpResult<Void> delete(@RequestBody List<Long> ids) {
        logger.info("[InfinityAccount] batch physical delete by ids = {}", ids);
        infinityAccountService.delete(ids);
        return HttpResult.ok();
    }

    @ApiOperation(value = "批量逻辑删除")
    @PutMapping(value = "/delete")
    public HttpResult<Void> updateDeleteFlag(@RequestBody List<Long> ids) {
        logger.info("[InfinityAccount] batch logic delete by ids = {}", ids);
        infinityAccountService.updateDeleteFlag(ids);
        return HttpResult.ok();
    }
}
