package com.elitesland.cbpl.infinity.server.api.controller;

import com.elitesland.cbpl.common.constant.ActiveStatus;
import com.elitesland.cbpl.infinity.server.api.service.InfinityApiService;
import com.elitesland.cbpl.infinity.server.api.vo.param.ApiQueryParamVO;
import com.elitesland.cbpl.infinity.server.api.vo.param.ApiSaveParamVO;
import com.elitesland.cbpl.infinity.server.api.vo.resp.InfinityApiDetailVO;
import com.elitesland.cbpl.infinity.server.api.vo.resp.InfinityApiRespVO;
import com.elitesland.cbpl.tool.api.domain.HttpResult;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;

/**
 * @author eric.hao
 * @since 2024/05/16
 */
@Slf4j
@Api(tags = "接口定义")
@RestController
@RequestMapping("/infinity/server/api")
@AllArgsConstructor
public class InfinityApiController {

    private final InfinityApiService apiService;

    @ApiOperation(value = "不分页查询")
    @PostMapping(value = "/query")
    public HttpResult<List<InfinityApiRespVO>> query(@RequestBody @Valid ApiQueryParamVO query) {
        logger.info("[InfinityApi] query list param = {}", query);
        return HttpResult.ok(apiService.infinityApiByParam(query));
    }

    @ApiOperation(value = "明细查询")
    @GetMapping(value = "/detail/{id}")
    public HttpResult<InfinityApiDetailVO> detail(@PathVariable("id") Long id) {
        logger.info("[InfinityApi] query detail by id = {}", id);
        return HttpResult.ok(apiService.infinityApiById(id));
    }

    @ApiOperation(value = "新增或修改")
    @PostMapping(value = "/save")
    public HttpResult<Long> save(@RequestBody @Valid ApiSaveParamVO saveParam) {
        logger.info("[InfinityApi] create/update record, saveParam = {}", saveParam);
        return HttpResult.ok(apiService.save(saveParam));
    }

    @ApiOperation(value = "启用")
    @PatchMapping(value = "/{id}/v")
    public HttpResult<Void> active(@PathVariable("id") Long id) {
        logger.info("[InfinityApi] active by id = {}", id);
        apiService.updateStatus(id, ActiveStatus.ACTIVE.getCode());
        return HttpResult.ok();
    }

    @ApiOperation(value = "禁用")
    @PatchMapping(value = "/{id}/x")
    public HttpResult<Void> inactive(@PathVariable("id") Long id) {
        logger.info("[InfinityApi] inactive by id = {}", id);
        apiService.updateStatus(id, ActiveStatus.INACTIVE.getCode());
        return HttpResult.ok();
    }

    @ApiOperation(value = "批量物理删除")
    @DeleteMapping(value = "/delete")
    public HttpResult<Long> delete(@RequestBody List<Long> ids) {
        logger.info("[InfinityApi] batch physical delete by ids = {}", ids);
        return HttpResult.ok(apiService.delete(ids));
    }
}
