package com.elitesland.cbpl.infinity.server.platform.convert;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.infinity.server.platform.entity.InfinityPlatformDO;
import com.elitesland.cbpl.infinity.server.platform.vo.param.PlatformSaveParamVO;
import com.elitesland.cbpl.infinity.server.platform.vo.resp.PlatformPagingVO;
import com.elitesland.cbpl.infinity.server.platform.vo.resp.PlatformRespVO;
import com.elitesland.cbpl.infinity.web.security.domain.BasicAuthAccountVO;
import com.elitesland.cbpl.infinity.web.security.domain.OAuth2AccountVO;
import com.elitesland.cbpl.tool.core.bean.BeanUtils;
import com.elitesland.cbpl.tool.db.BaseMapperConfig;
import org.mapstruct.Mapper;
import org.mapstruct.Mapping;
import org.mapstruct.MappingTarget;
import org.mapstruct.Named;
import org.mapstruct.factory.Mappers;

import java.util.List;

import static com.elitesland.cbpl.common.constant.AuthMethod.AUTH_METHOD_BASIC_AUTH;
import static com.elitesland.cbpl.common.constant.AuthMethod.AUTH_METHOD_OAUTH_V2;

/**
 * 实体对象转换类
 *
 * @author eric.hao
 * @since 2024/05/16
 */
@Mapper(config = BaseMapperConfig.class)
public interface InfinityPlatformConvert {

    InfinityPlatformConvert INSTANCE = Mappers.getMapper(InfinityPlatformConvert.class);

    /**
     * DO => PageVO
     */
    List<PlatformPagingVO> toPageVO(List<InfinityPlatformDO> platformDO);

    /**
     * DO => PageVO
     */
    @Mapping(target = "basicAuthAccount", expression = "java(toBasicAuthAccount(platformDO))")
    @Mapping(target = "oauth2Account", expression = "java(toOAuthAccount(platformDO))")
    PlatformPagingVO toPageVO(InfinityPlatformDO platformDO);

    /**
     * DO => RespVO
     */
    List<PlatformRespVO> doToVO(List<InfinityPlatformDO> platformDO);

    /**
     * DO => RespVO
     */
    @Mapping(target = "basicAuthAccount", expression = "java(toBasicAuthAccount(platformDO))")
    @Mapping(target = "oauth2Account", expression = "java(toOAuthAccount(platformDO))")
    PlatformRespVO doToVO(InfinityPlatformDO platformDO);

    /**
     * SaveParam => DO
     */
    @Mapping(target = "authAccount", expression = "java(authAccountToStr(saveParam))")
    InfinityPlatformDO saveParamToDO(PlatformSaveParamVO saveParam);

    /**
     * SaveParam Merge-Into DO
     */
    @Mapping(target = "authAccount", expression = "java(authAccountToStr(saveParam))")
    void saveParamMergeToDO(PlatformSaveParamVO saveParam, @MappingTarget InfinityPlatformDO infinityPlatformDO);

    @Named("authAccountToStr")
    default String authAccountToStr(PlatformSaveParamVO saveParam) {
        if (saveParam.getAuthMethod().equals(AUTH_METHOD_BASIC_AUTH.getCode())) {
            BasicAuthAccountVO account = saveParam.getBasicAuthAccount();
            if (ObjectUtil.isNull(account) || account.isEmpty()) {
                throw new RuntimeException("account is empty.");
            }
            return BeanUtils.toJsonStr(account);
        }
        if (saveParam.getAuthMethod().equals(AUTH_METHOD_OAUTH_V2.getCode())) {
            OAuth2AccountVO account = saveParam.getOauth2Account();
            if (ObjectUtil.isNull(account) || account.isEmpty()) {
                throw new RuntimeException("account is empty.");
            }
            return BeanUtils.toJsonStr(account);
        }
        return "";
    }

    @Named("toBasicAuthAccount")
    default BasicAuthAccountVO toBasicAuthAccount(InfinityPlatformDO platformDO) {
        if (StrUtil.isBlank(platformDO.getAuthAccount())) {
            return null;
        }
        if (platformDO.getAuthMethod().equals(AUTH_METHOD_BASIC_AUTH.getCode())) {
            return BeanUtils.toBean(platformDO.getAuthAccount(), BasicAuthAccountVO.class);
        }
        return null;
    }

    @Named("toOAuthAccount")
    default OAuth2AccountVO toOAuthAccount(InfinityPlatformDO platformDO) {
        if (StrUtil.isBlank(platformDO.getAuthAccount())) {
            return null;
        }
        if (platformDO.getAuthMethod().equals(AUTH_METHOD_OAUTH_V2.getCode())) {
            return BeanUtils.toBean(platformDO.getAuthAccount(), OAuth2AccountVO.class);
        }
        return null;
    }
}
