package com.elitesland.cbpl.infinity.web.config;

import com.elitesland.cbpl.infinity.web.http.service.InfinityRestClient;
import com.elitesland.cbpl.infinity.web.security.service.oauth2.OAuth2Client;
import io.netty.channel.ChannelOption;
import io.netty.handler.timeout.ReadTimeoutHandler;
import io.netty.handler.timeout.WriteTimeoutHandler;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.client.reactive.ReactorClientHttpConnector;
import org.springframework.web.reactive.function.client.WebClient;
import reactor.netty.http.client.HttpClient;

import java.time.Duration;
import java.util.concurrent.TimeUnit;

import static com.elitesland.cbpl.infinity.web.config.InfinityWebProperties.INFINITY_WEB_CONFIG_PREFIX;

/**
 * @author eric.hao
 * @since 2021/07/13
 */
@Slf4j
@Configuration
@EnableConfigurationProperties(InfinityWebProperties.class)
@ConditionalOnProperty(prefix = INFINITY_WEB_CONFIG_PREFIX, name = "enabled", havingValue = "true")
@RequiredArgsConstructor
public class InfinityWebAutoConfiguration {

    @Bean
    @ConditionalOnMissingBean
    public WebClient webClient() {
        HttpClient httpClient = HttpClient.create()
                // 连接超时
                .option(ChannelOption.CONNECT_TIMEOUT_MILLIS, 5000)
                // 响应超时
                .responseTimeout(Duration.ofMillis(5000))
                .doOnConnected(conn -> conn
                        // 读写超时
                        .addHandlerLast(new ReadTimeoutHandler(5000, TimeUnit.MILLISECONDS))
                        .addHandlerLast(new WriteTimeoutHandler(5000, TimeUnit.MILLISECONDS))
                );
        return WebClient.builder().clientConnector(new ReactorClientHttpConnector(httpClient)).build();
    }

    @Bean
    public InfinityRestClient infinityRestClient(WebClient webClient) {
        return new InfinityRestClient(webClient);
    }

    @Bean
    public OAuth2Client oAuth2Client(WebClient webClient) {
        return new OAuth2Client(webClient);
    }
}
