package com.elitesland.cbpl.infinity.web.http.domain;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.date.LocalDateTimeUtil;
import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.data.etl.transform.util.TransformUtil;
import com.elitesland.cbpl.infinity.common.constant.SyncConstant;
import com.elitesland.cbpl.infinity.server.api.vo.resp.InfinityApiDetailVO;
import com.elitesland.cbpl.infinity.web.http.param.HttpParam;
import com.elitesland.cbpl.logging.infinity.domain.InfinityLogVO;
import com.elitesland.cbpl.logging.infinity.util.InfinityLogUtil;
import com.elitesland.cbpl.tool.core.exceptions.ExceptionUtils;
import com.elitesland.cbpl.tool.core.http.HttpServletUtil;
import com.elitesland.cbpl.tool.tenant.TenantSpiUtil;
import com.lzhpo.tracer.util.TracerUtils;
import io.swagger.annotations.ApiModelProperty;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.web.reactive.function.client.ClientResponse;
import org.springframework.web.reactive.function.client.WebClient;
import reactor.core.publisher.Mono;

import java.time.LocalDateTime;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;

import static cn.hutool.core.date.DatePattern.NORM_DATETIME_MS_PATTERN;

/**
 * @author eric.hao
 * @since 2024/05/20
 */
@Slf4j
@AllArgsConstructor(staticName = "of")
public class InfinityResult<T> {

    @ApiModelProperty(value = "响应值的实体类型")
    private Class<T> responseType;

    private HttpParam param;

    private InfinityLogVO logVO;

    private WebClient.RequestBodySpec uri;

    public List<T> toList() {
        return wrapper(response -> response.bodyToFlux(responseType).collectList());
    }

    public T toEntity() {
        return wrapper(response -> response.bodyToMono(responseType));
    }

    /**
     * 重构接口日志的报文结构
     *
     * @since 0.3.1-SNAPSHOT
     */
    private <R> R wrapper(Function<ClientResponse, Mono<R>> fn) {
        LocalDateTime startTime = LocalDateTime.now();
        AtomicReference<HttpStatus> httpStatus = new AtomicReference<>(null);
        AtomicReference<String> errorMessage = new AtomicReference<>("");
        R result = uri.exchangeToMono(response -> {
            httpStatus.set(response.statusCode());
            if (response.statusCode() == HttpStatus.OK) {
                return fn.apply(response);
            }
            // Log error and retry logic
            return response.createException().flatMap(Mono::error);
        }).onErrorResume(e -> {
            errorMessage.set(ExceptionUtils.formatException(e, SyncConstant.ERR_EXCEED_LENGTH));
            logger.error("[INFINITY][{}] 接口请求失败: ", param.getApiCode(), e);
            // Handle error silently without throwing exception
            return Mono.empty();
        }).block();
        LocalDateTime endTime = LocalDateTime.now();
        boolean success = httpStatus.get() == HttpStatus.OK;
        InfinityLogUtil.record(toInfinityParam(result, success, errorMessage.get(), startTime, endTime));
        // Throw Exception
        if (!success) {
            String defaultMessage = "[" + param.getApiCode() + "] 接口调用异常：" + httpStatus.get().value();
            throw new RuntimeException(StrUtil.blankToDefault(errorMessage.get(), defaultMessage));
        }
        return result;
    }

    /**
     * 报文转换：Api Result => InfinityLogVO
     *
     * @since 0.3.1-SNAPSHOT
     */
    private InfinityLogVO toInfinityParam(Object responseBody, boolean requestSuccess, String reqFailMsg, LocalDateTime startTime, LocalDateTime endTime) {
        // 接口定义
        InfinityApiDetailVO apiConfig = param.getApiConfig();
        logVO.setAddressIp(HttpServletUtil.currentClientIp());
        logVO.setRequestTime(DateUtil.format(startTime, NORM_DATETIME_MS_PATTERN));
        logVO.setResponseTime(DateUtil.format(endTime, NORM_DATETIME_MS_PATTERN));
        long spendTime = LocalDateTimeUtil.between(startTime, endTime).toMillis();
        logVO.setSpendTime(spendTime);
        logVO.setRequestSuccess(requestSuccess);
        logVO.setRequestFailMessage(reqFailMsg);
        logVO.setResponseBody(responseBody);
        logVO.setTraceId(TracerUtils.getTraceId());
        logVO.setTenantCode(TenantSpiUtil.currentTenantCode());
        logVO.setTargetApp(apiConfig.getPlatformCode());
        logVO.setBusinessType(apiConfig.getApiCode());
        logVO.setBusinessKey(param.getBizKey());

        // 成功状态值
        String successStatus = apiConfig.getResponseSuccessStatus();
        // 响应状态值
        String responseStatus = TransformUtil.parse(responseBody, apiConfig.getResponseSuccessSpel());
        // 接口返回状态、接口定义状态，都不为空
        if (StrUtil.isNotBlank(responseStatus) && StrUtil.isNotBlank(successStatus)) {
            logVO.setResponseSuccess(responseStatus.equals(successStatus));
            logVO.setResponseFailMessage(TransformUtil.parse(responseBody, apiConfig.getResponseErrorMessageSpel()));
        }
        // 任一为空，则以 request 状态码为准
        else {
            logVO.setResponseSuccess(requestSuccess);
            logVO.setResponseFailMessage(reqFailMsg);
        }
        return logVO;
    }
}
