package com.elitesland.cbpl.infinity.server.api.repo;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.infinity.server.api.entity.QInfinityApiDO;
import com.elitesland.cbpl.infinity.server.api.vo.param.ApiQueryParamVO;
import com.elitesland.cbpl.infinity.server.api.vo.resp.InfinityApiMergeVO;
import com.elitesland.cbpl.infinity.server.api.vo.resp.InfinityApiRespVO;
import com.elitesland.cbpl.infinity.server.folder.entity.QInfinityFolderDO;
import com.elitesland.cbpl.infinity.server.platform.entity.QInfinityPlatformDO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import com.querydsl.jpa.impl.JPADeleteClause;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

/**
 * @author eric.hao
 * @since 2024/05/16
 */
@Component
@RequiredArgsConstructor
public class InfinityApiRepoProc {

    private final JPAQueryFactory jpaQueryFactory;
    private static final QInfinityApiDO infinityApiDO = QInfinityApiDO.infinityApiDO;
    private static final QInfinityFolderDO folderDO = QInfinityFolderDO.infinityFolderDO;
    private static final QInfinityPlatformDO platformDO = QInfinityPlatformDO.infinityPlatformDO;

    private final QBean<InfinityApiRespVO> infinityApiVO = Projections.bean(
            InfinityApiRespVO.class,
            infinityApiDO.id,
            infinityApiDO.platformCode,
            infinityApiDO.folderId,
            folderDO.folderName,
            infinityApiDO.serverUrl,
            infinityApiDO.authMethod,
            infinityApiDO.authAccount,
            infinityApiDO.apiCode,
            infinityApiDO.apiName,
            infinityApiDO.apiUrl,
            infinityApiDO.requestMethod,
            infinityApiDO.apiStatus,
            infinityApiDO.headerParam,
            infinityApiDO.queryParam,
            infinityApiDO.pathParam,
            infinityApiDO.bodyParam,
            infinityApiDO.response,
            infinityApiDO.description,
            infinityApiDO.remark,
            infinityApiDO.createUserId,
            infinityApiDO.creator,
            infinityApiDO.createTime,
            infinityApiDO.modifyUserId,
            infinityApiDO.updater,
            infinityApiDO.modifyTime,
            infinityApiDO.deleteFlag
    );

    private Predicate where(ApiQueryParamVO query) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(infinityApiDO.deleteFlag.eq(0));
        if (StrUtil.isNotBlank(query.getPlatformCode())) {
            predicates.add(infinityApiDO.platformCode.eq(query.getPlatformCode()));
        }
//        if (ObjectUtil.isNotNull(query.getFolderId())) {
//            predicates.add(infinityApiDO.folderId.eq(query.getFolderId()));
//        }
//        if (StrUtil.isNotBlank(query.getApiCodeName())) {
//            String likeStr = SqlUtil.toSqlLikeString(query.getApiCodeName());
//            predicates.add(infinityApiDO.apiCode.eq(likeStr).or(infinityApiDO.apiName.like(likeStr)));
//        }
//        if (ObjectUtil.isNotNull(query.getApiStatus())) {
//            predicates.add(infinityApiDO.apiStatus.eq(query.getApiStatus()));
//        }
        return ExpressionUtils.allOf(predicates);
    }

    public List<InfinityApiRespVO> infinityApiByParam(ApiQueryParamVO query) {
        var jpaQuery = jpaQueryFactory.select(infinityApiVO)
                .from(infinityApiDO)
                .leftJoin(folderDO).on(folderDO.id.eq(infinityApiDO.folderId));
        jpaQuery.where(this.where(query));
        return jpaQuery.fetch();
    }

    public long delete(List<Long> ids) {
        JPADeleteClause delete = jpaQueryFactory.delete(infinityApiDO)
                .where(infinityApiDO.id.in(ids));
        return delete.execute();
    }

    public long updateDeleteFlag(List<Long> ids) {
        JPAUpdateClause update = jpaQueryFactory.update(infinityApiDO)
                .set(infinityApiDO.deleteFlag, 1)
                .where(infinityApiDO.id.in(ids));
        return update.execute();
    }

    public long updateStatus(Long id, int status) {
        JPAUpdateClause update = jpaQueryFactory.update(infinityApiDO)
                .set(infinityApiDO.apiStatus, status)
                .where(infinityApiDO.id.eq(id));
        return update.execute();
    }

    public InfinityApiMergeVO findByApiCode(String apiCode) {
        if (StrUtil.isBlank(apiCode)) {
            return null;
        }
        var infinityApiVO = Projections.bean(
                InfinityApiMergeVO.class,
                infinityApiDO.id,
                infinityApiDO.platformCode,
                folderDO.id.as("folderId"),
                infinityApiDO.serverUrl.as("apiServerUrl"),
                infinityApiDO.authMethod.as("apiAuthMethod"),
                infinityApiDO.authAccount.as("apiAuthAccount"),
                folderDO.serverUrl.as("folderServerUrl"),
                folderDO.authMethod.as("folderAuthMethod"),
                folderDO.authAccount.as("folderAuthAccount"),
                platformDO.serverUrl.as("platformServerUrl"),
                platformDO.authMethod.as("platformAuthMethod"),
                platformDO.authAccount.as("platformAuthAccount"),
                infinityApiDO.apiCode,
                infinityApiDO.apiName,
                infinityApiDO.apiUrl,
                infinityApiDO.requestMethod,
                infinityApiDO.apiStatus,
                infinityApiDO.headerParam,
                infinityApiDO.queryParam,
                infinityApiDO.pathParam,
                infinityApiDO.mediaType,
                infinityApiDO.bodyParam,
                infinityApiDO.response,
                infinityApiDO.responseSuccessSpel,
                infinityApiDO.responseSuccessStatus,
                infinityApiDO.responseErrorMessageSpel,
                infinityApiDO.description,
                infinityApiDO.remark,
                infinityApiDO.createUserId,
                infinityApiDO.creator,
                infinityApiDO.createTime,
                infinityApiDO.modifyUserId,
                infinityApiDO.updater,
                infinityApiDO.modifyTime,
                infinityApiDO.deleteFlag
        );
        var jpaQuery = jpaQueryFactory.select(infinityApiVO)
                .from(infinityApiDO)
                .leftJoin(folderDO).on(folderDO.id.eq(infinityApiDO.folderId))
                .leftJoin(platformDO).on(platformDO.platformCode.eq(infinityApiDO.platformCode));
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(infinityApiDO.apiCode.eq(apiCode));
        jpaQuery.where(ExpressionUtils.allOf(predicates));
        return jpaQuery.fetchOne();
    }

    /**
     * 判断接口编码，是否已存在
     */
    public boolean existsByCode(Long id, String apiCode) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(infinityApiDO.apiCode.eq(apiCode));
        if (ObjectUtil.isNotNull(id)) {
            predicates.add(infinityApiDO.id.ne(id));
        }
        var jpaQuery = jpaQueryFactory.select(infinityApiDO.id)
                .from(infinityApiDO);
        jpaQuery.where(ExpressionUtils.allOf(predicates));
        return !jpaQuery.fetch().isEmpty();
    }
}
