package com.elitesland.cbpl.infinity.web.security.service.oauth2;

import cn.hutool.core.collection.ListUtil;
import cn.hutool.core.text.StrPool;
import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.logging.syslog.util.LogUtil;
import com.elitesland.cbpl.tool.core.bean.BeanUtils;
import com.elitesland.cbpl.tool.core.exceptions.ExceptionUtils;
import com.elitesland.cbpl.tool.core.http.RestWrapper;
import com.elitesland.cbpl.tool.redis.util.RedisUtil;
import com.elitesland.cbpl.tool.tenant.TenantSpiUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * @author eric.hao
 * @since 2024/06/07
 */
@Slf4j
@RequiredArgsConstructor
public class OAuth2Client {

    //    private final WebClient webClient;
    private final RestWrapper restWrapper;
    private final RedisUtil redisUtil;


    /**
     * 获取可用的token
     *
     * @param platformCode 平台代码
     * @param url          认证地址
     * @param clientId     账号
     * @param clientSecret 密码
     * @return token
     */
    public String getAccessToken(String platformCode, String url, String clientId, String clientSecret) {
        String tokenKey = tokenKey(platformCode, clientId);
        String cacheToken = redisUtil.getString(tokenKey);
        if (StrUtil.isNotBlank(cacheToken)) {
            LogUtil.info("[BPMN][INFINITY] CacheToken(" + tokenKey + "): ", cacheToken);
            return cacheToken;
        }
        return createAccessToken(platformCode, url, clientId, clientSecret);
    }

    /**
     * 生成新的token
     *
     * @param platformCode 平台代码
     * @param url          认证地址
     * @param clientId     账号
     * @param clientSecret 密码
     * @return token
     */
    private String createAccessToken(String platformCode, String url, String clientId, String clientSecret) {
        AccessToken token = authentication(url, clientId, clientSecret, AccessToken.class);
        String accessToken = token.getAccessToken();
        Long expiresIn = token.getExpiresIn();
        LogUtil.info("[BPMN][INFINITY] AccessToken: ", accessToken);
        LogUtil.info("[BPMN][INFINITY] ExpiresIn: ", expiresIn);
        if (expiresIn > 10) {
            redisUtil.set(tokenKey(platformCode, clientId), accessToken, expiresIn - 5, TimeUnit.SECONDS);
        }
        return accessToken;
    }

    public String authentication(String url, String clientId, String clientSecret) {
        String accessToken = authentication(url, clientId, clientSecret, String.class);
        LogUtil.info("[BPMN][INFINITY] AccessToken: ", accessToken);
        return accessToken;
    }

//    public <T> T authentication(String url, String clientId, String clientSecret, Class<T> clazz) {
//        LogUtil.info("[BPMN][INFINITY] OAuth2 Url: " + url);
//        LogUtil.info("[BPMN][INFINITY] OAuth2 clientId: " + clientId);
//        var uriSpec = webClient.method(HttpMethod.POST);
//        var bodySpec = uriSpec.uri(url);
//        bodySpec.acceptCharset(StandardCharsets.UTF_8);
//        bodySpec.contentType(MediaType.APPLICATION_FORM_URLENCODED);
//        MultiValueMap<String, Object> multiValueMap = new LinkedMultiValueMap<>();
//        multiValueMap.add("grant_type", "client_credentials");
//        multiValueMap.add("client_id", clientId);
//        multiValueMap.add("client_secret", clientSecret);
//        bodySpec.body(BodyInserters.fromMultipartData(multiValueMap));
//        return bodySpec.retrieve().bodyToMono(clazz).block();
//    }

    /**
     * TODO 暂时的解决方案 待完善，使用相对简单的对接方式
     */
    public <T> T authentication(String url, String clientId, String clientSecret, Class<T> clazz) {
        LogUtil.info("[BPMN][INFINITY] Url: ", url);
        LogUtil.info("[BPMN][INFINITY] ClientId: ", clientId);
        try {
            Map<String, Object> param = new HashMap<>();
            param.put("grant_type", "client_credentials");
            param.put("client_id", clientId);
            param.put("client_secret", clientSecret);
            String result = restWrapper.postFormUrlencoded(url, param);
            return BeanUtils.toBean(result, clazz);
        } catch (Exception e) {
            LogUtil.error("[BPMN][INFINITY] 认证失败：", ExceptionUtils.formatException(e));
            throw new RuntimeException(e);
        }
    }

    /**
     * Redis TokenKey
     *
     * @param platformCode 平台编码
     * @param clientId     账号ID
     * @return tokenKey
     */
    private String tokenKey(String platformCode, String clientId) {
        List<String> values = ListUtil.toList(platformCode, clientId);
        String tenantCode = TenantSpiUtil.currentTenantCode();
        if (StrUtil.isNotBlank(tenantCode)) {
            values.add(tenantCode);
        }
        return StrUtil.join(StrPool.UNDERLINE, values);
    }
}
