package com.elitesland.cbpl.infinity.web.security.service.oauth2;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.common.constant.AuthMethod;
import com.elitesland.cbpl.infinity.web.http.param.HttpParam;
import com.elitesland.cbpl.infinity.web.security.domain.OAuth2AccountVO;
import com.elitesland.cbpl.infinity.web.security.service.SecurityAuthService;
import com.elitesland.cbpl.logging.syslog.util.LogUtil;
import com.elitesland.cbpl.unicom.annotation.UnicomTag;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.stereotype.Service;
import org.springframework.web.reactive.function.client.WebClient;

import static com.elitesland.cbpl.infinity.web.config.InfinityWebProperties.INFINITY_WEB_ENABLED;

/**
 * `@UnicomTag` 参考 {@link AuthMethod}
 *
 * @author eric.hao
 * @since 2024/06/07
 */
@Slf4j
@Service
@RequiredArgsConstructor
@UnicomTag("OAUTH_2.0")
public class OAuth2SecurityService implements SecurityAuthService {

    @Autowired(required = false)
    private OAuth2Client authClient;

    @Override
    public boolean authentication(WebClient.RequestBodySpec uri, HttpParam param) {
        LogUtil.info("[BPMN][INFINITY] 认证方式: " + AuthMethod.AUTH_METHOD_OAUTH_V2.getDesc());
        if (!INFINITY_WEB_ENABLED) {
            LogUtil.error("[BPMN][INFINITY] WebClient not enabled.");
            throw new RuntimeException("WebClient not enabled.");
        }
        OAuth2AccountVO account = param.getApiConfig().getOauth2Account();
        if (ObjectUtil.isNull(account) || account.isEmpty()) {
            LogUtil.error("[BPMN][INFINITY] 未维护认证账号.");
            throw new RuntimeException("account is empty.");
        }
        String accessToken = authClient.getAccessToken(param.getPlatformCode(),
                account.getUrl(), account.getClientId(), account.getClientSecret());
        if (StrUtil.isBlank(accessToken)) {
            LogUtil.error("[BPMN][INFINITY] token为空.");
            return false;
        }
        uri.header(HttpHeaders.AUTHORIZATION, "Bearer " + accessToken);
        return true;
    }
}
