package com.elitesland.cbpl.infinity.web.http.domain;

import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.infinity.common.constant.SyncConstant;
import com.elitesland.cbpl.infinity.web.http.param.HttpParam;
import com.elitesland.cbpl.logging.infinity.util.InfinityLogUtil;
import com.elitesland.cbpl.tool.core.exceptions.ExceptionUtils;
import io.swagger.annotations.ApiModelProperty;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.util.StopWatch;
import org.springframework.web.reactive.function.client.ClientResponse;
import org.springframework.web.reactive.function.client.WebClient;
import reactor.core.publisher.Mono;

import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;

/**
 * @author eric.hao
 * @since 2024/05/20
 */
@Slf4j
@AllArgsConstructor(staticName = "of")
public class InfinityResult<T> {

    @ApiModelProperty(value = "响应值的实体类型")
    private Class<T> responseType;

    private HttpParam param;

    private WebClient.RequestBodySpec uri;

    public List<T> toList() {
        return wrapper(response -> response.bodyToFlux(responseType).collectList());
    }

    public T toEntity() {
        return wrapper(response -> response.bodyToMono(responseType));
    }

    private <R> R wrapper(Function<ClientResponse, Mono<R>> fn) {
        StopWatch stopWatch = new StopWatch();
        stopWatch.start();
        AtomicReference<String> errorMessage = new AtomicReference<>("");
        R result = uri.exchangeToMono(response -> {
            if (response.statusCode() == HttpStatus.OK) {
                return fn.apply(response);
            }
            // Log error and retry logic
            return response.createException().flatMap(Mono::error);
        }).onErrorResume(e -> {
            errorMessage.set(ExceptionUtils.formatException(e, SyncConstant.ERR_EXCEED_LENGTH));
            logger.error("[INFINITY] RestAPI Invoke Error: ", e);
            // Handle error silently without throwing exception
            return Mono.empty();
        }).block();
        stopWatch.stop();
        InfinityLogUtil.record(param, result, errorMessage.get(), stopWatch.getTotalTimeSeconds());
        // Throw Exception
        if (StrUtil.isNotBlank(errorMessage.get())) {
            throw new RuntimeException(errorMessage.get());
        }
        return result;
    }
}
