package com.elitesland.cbpl.infinity.server.api.service.impl;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import com.elitesland.cbpl.infinity.server.api.convert.InfinityApiConvert;
import com.elitesland.cbpl.infinity.server.api.entity.InfinityApiDO;
import com.elitesland.cbpl.infinity.server.api.repo.InfinityApiRepo;
import com.elitesland.cbpl.infinity.server.api.repo.InfinityApiRepoProc;
import com.elitesland.cbpl.infinity.server.api.service.InfinityApiService;
import com.elitesland.cbpl.infinity.server.api.vo.param.ApiQueryParamVO;
import com.elitesland.cbpl.infinity.server.api.vo.param.ApiSaveParamVO;
import com.elitesland.cbpl.infinity.server.api.vo.resp.InfinityApiDetailVO;
import com.elitesland.cbpl.infinity.server.api.vo.resp.InfinityApiMergeVO;
import com.elitesland.cbpl.infinity.server.api.vo.resp.InfinityApiRespVO;
import com.elitesland.cbpl.infinity.server.folder.convert.InfinityFolderConvert;
import com.elitesland.cbpl.infinity.server.folder.entity.InfinityFolderDO;
import com.elitesland.cbpl.infinity.server.folder.repo.InfinityFolderRepoProc;
import com.elitesland.cbpl.tool.core.bean.BeanUtils;
import com.elitesland.cbpl.tool.core.exceptions.PhoenixException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author eric.hao
 * @since 2024/05/16
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class InfinityApiServiceImpl implements InfinityApiService {

    private final InfinityApiRepo infinityApiRepo;
    private final InfinityApiRepoProc infinityApiRepoProc;
    private final InfinityFolderRepoProc folderRepoProc;

    @Override
    public List<InfinityApiRespVO> infinityApiByParam(ApiQueryParamVO query) {
        // 所有接口
        List<InfinityApiRespVO> responseVO = infinityApiRepoProc.infinityApiByParam(query);
        // 所有目录
        Map<Long, InfinityApiRespVO> folderMap = queryFolderByParam(query);
        // 按目录分组
        Map<Long, List<InfinityApiRespVO>> apiMap = responseVO.stream()
                .filter(row -> row.getFolderId() != null)
                .collect(Collectors.groupingBy(InfinityApiRespVO::getFolderId));
        responseVO.removeIf(row -> row.getFolderId() != null);
        // 配置目录下的接口列表
        apiMap.forEach((folderId, children) -> {
            var curr = BeanUtils.toBean(folderMap.get(folderId), InfinityApiRespVO.class);
            children.sort(InfinityApiRespVO::compareTo);
            curr.setChildren(children);
            responseVO.add(curr);
            // 排除已添加的目录
            folderMap.remove(folderId);
        });
        // 追加未添加的目录
        folderMap.forEach((folderId, api) -> responseVO.add(api));
        // 按名称，对目录/接口排序
        responseVO.sort(InfinityApiRespVO::compareTo);
        return responseVO;
    }

    /**
     * 查询该平台下，所有分组
     */
    private Map<Long, InfinityApiRespVO> queryFolderByParam(ApiQueryParamVO query) {
        List<InfinityFolderDO> folderRespVOS = folderRepoProc.infinityFolderByParam(query);
        List<InfinityApiRespVO> folders = InfinityFolderConvert.INSTANCE.toApiFolder(folderRespVOS);
        return folders.stream().collect(Collectors.toMap(InfinityApiRespVO::getFolderId, Function.identity()));
    }

    @Override
    public InfinityApiDetailVO infinityApiById(Long id) {
        Optional<InfinityApiDO> infinityApiDO = infinityApiRepo.findById(id);
        if (infinityApiDO.isEmpty()) {
            throw PhoenixException.unexpected("Not Found Data");
        }
        return InfinityApiConvert.INSTANCE.doToVO(infinityApiDO.get());
    }

    @Override
    public InfinityApiDetailVO infinityApiByApiCode(String apiCode) {
        InfinityApiMergeVO apiMergeVO = infinityApiRepoProc.findByApiCode(apiCode);
        if (ObjectUtil.isNull(apiMergeVO)) {
            throw PhoenixException.unexpected("Not Found Data");
        }
        return InfinityApiConvert.INSTANCE.mergeToDetailVO(apiMergeVO);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public Long save(ApiSaveParamVO saveParam) {
        boolean exists = infinityApiRepoProc.existsByCode(saveParam.getId(), saveParam.getApiCode());
        Assert.isFalse(exists, "编码已存在");
        // TODO 参数规则检查
        // 新增
        if (saveParam.isNew()) {
            InfinityApiDO infinityApiDO = InfinityApiConvert.INSTANCE.saveParamToDO(saveParam);
            infinityApiRepo.save(infinityApiDO);
            return infinityApiDO.getId();
        }
        // 修改
        else {
            Optional<InfinityApiDO> infinityApiDO = infinityApiRepo.findById(saveParam.getId());
            if (infinityApiDO.isEmpty()) {
                throw PhoenixException.unexpected("Not Found Data");
            }
            InfinityApiDO infinityApi = infinityApiDO.get();
            InfinityApiConvert.INSTANCE.saveParamMergeToDO(saveParam, infinityApi);
            infinityApiRepo.save(infinityApi);
            return infinityApi.getId();
        }
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void updateStatus(Long id, int status) {
        infinityApiRepoProc.updateStatus(id, status);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public long delete(List<Long> ids) {
        return infinityApiRepoProc.delete(ids);
    }
}
