package com.elitesland.cbpl.infinity.server.platform.service.impl;

import cn.hutool.core.lang.Assert;
import com.elitesland.cbpl.infinity.common.service.AuthMethodService;
import com.elitesland.cbpl.infinity.server.platform.convert.InfinityPlatformConvert;
import com.elitesland.cbpl.infinity.server.platform.entity.InfinityPlatformDO;
import com.elitesland.cbpl.infinity.server.platform.repo.InfinityPlatformRepo;
import com.elitesland.cbpl.infinity.server.platform.repo.InfinityPlatformRepoProc;
import com.elitesland.cbpl.infinity.server.platform.service.InfinityPlatformService;
import com.elitesland.cbpl.infinity.server.platform.vo.param.PlatformPagingParamVO;
import com.elitesland.cbpl.infinity.server.platform.vo.param.PlatformQueryParamVO;
import com.elitesland.cbpl.infinity.server.platform.vo.param.PlatformSaveParamVO;
import com.elitesland.cbpl.infinity.server.platform.vo.resp.PlatformPagingVO;
import com.elitesland.cbpl.infinity.server.platform.vo.resp.PlatformRespVO;
import com.elitesland.cbpl.tool.core.exceptions.PhoenixException;
import com.elitesland.cbpl.tool.db.PagingVO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @author eric.hao
 * @since 2024/05/16
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class InfinityPlatformServiceImpl implements InfinityPlatformService {

    private final AuthMethodService authMethodService;
    private final InfinityPlatformRepo platformRepo;
    private final InfinityPlatformRepoProc platformRepoProc;

    @Override
    public PagingVO<PlatformPagingVO> platformPageBy(PlatformPagingParamVO query) {
        long count = platformRepoProc.platformCountBy(query);
        if (count > 0) {
            var list = platformRepoProc.platformPageBy(query);
            return new PagingVO<>(count, InfinityPlatformConvert.INSTANCE.toPageVO(list));
        }
        return new PagingVO<>();
    }

    @Override
    public List<PlatformRespVO> platformByParam(PlatformQueryParamVO query) {
        List<InfinityPlatformDO> list = platformRepoProc.platformByParam(query);
        List<PlatformRespVO> platforms = InfinityPlatformConvert.INSTANCE.doToVO(list);
        List<String> authMethodCodes = platforms.stream().map(PlatformRespVO::getAuthMethod).collect(Collectors.toList());
        Map<String, String> authMethodNames = authMethodService.fromCode(authMethodCodes);
        for (var platform : platforms) {
            String authMethodName = authMethodNames.getOrDefault(platform.getAuthMethod(), platform.getAuthMethod());
            platform.setAuthMethodName(authMethodName);
        }
        return platforms;
    }

    @Override
    public PlatformRespVO platformById(Long id) {
        Optional<InfinityPlatformDO> infinityPlatformDO = platformRepo.findById(id);
        if (infinityPlatformDO.isEmpty()) {
            throw PhoenixException.unexpected("Not Found Data");
        }
        return InfinityPlatformConvert.INSTANCE.doToVO(infinityPlatformDO.get());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public Long save(PlatformSaveParamVO saveParam) {
        boolean exists = platformRepoProc.existsByCode(saveParam.getId(), saveParam.getPlatformCode());
        Assert.isFalse(exists, "编码已存在");
        // 新增
        if (saveParam.isNew()) {
            InfinityPlatformDO infinityPlatformDO = InfinityPlatformConvert.INSTANCE.saveParamToDO(saveParam);
            platformRepo.save(infinityPlatformDO);
            return infinityPlatformDO.getId();
        }
        // 修改
        else {
            Optional<InfinityPlatformDO> infinityPlatformDO = platformRepo.findById(saveParam.getId());
            if (infinityPlatformDO.isEmpty()) {
                throw PhoenixException.unexpected("Not Found Data");
            }
            InfinityPlatformDO infinityPlatform = infinityPlatformDO.get();
            InfinityPlatformConvert.INSTANCE.saveParamMergeToDO(saveParam, infinityPlatform);
            platformRepo.save(infinityPlatform);
            return infinityPlatform.getId();
        }
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public long delete(List<Long> ids) {
        // TODO 删除该平台下，所有目录及接口
        return platformRepoProc.delete(ids);
    }
}
