package com.elitesland.cbpl.infinity.client.security.interceptor;

import cn.hutool.core.util.CharsetUtil;
import cn.hutool.http.HttpStatus;
import com.elitesland.cbpl.infinity.client.security.spi.InfinityAuthListener;
import com.elitesland.cbpl.infinity.client.security.session.InfinitySession;
import com.elitesland.cbpl.infinity.server.account.vo.resp.InfinityAccountRespVO;
import com.elitesland.cbpl.logging.syslog.util.LogUtil;
import com.elitesland.cbpl.tool.api.domain.HttpResult;
import com.elitesland.cbpl.tool.core.bean.BeanUtils;
import com.elitesland.cbpl.tool.crypto.BasicAuthUtil;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.protocol.HTTP;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;

import javax.annotation.Nonnull;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.List;
import java.util.Optional;

import static org.springframework.http.MediaType.APPLICATION_JSON_VALUE;

/**
 * @author eric.hao
 * @since 2023/05/30
 */
@Slf4j
@Data
@NoArgsConstructor
public class AuthInterceptor implements HandlerInterceptor {

    private List<InfinityAccountRespVO> accounts;
    private InfinityAuthListener infinityAuthListener;

    public AuthInterceptor(List<InfinityAccountRespVO> accounts) {
        this.accounts = accounts;
    }

    public AuthInterceptor(List<InfinityAccountRespVO> accounts, InfinityAuthListener infinityAuthListener) {
        this.accounts = accounts;
        this.infinityAuthListener = infinityAuthListener;
    }

    @Override
    public boolean preHandle(@Nonnull HttpServletRequest request, @Nonnull HttpServletResponse response, @Nonnull Object handler) throws IOException {
        response.addHeader(HTTP.CONTENT_TYPE, APPLICATION_JSON_VALUE);
        response.setCharacterEncoding(CharsetUtil.UTF_8);
        try {
            if (infinityAuthListener != null) {
                infinityAuthListener.preHandle(request);
            }
            // get header token
            final String token = request.getHeader("Authorization");
            LogUtil.info("", "[INFINITY] 读取请求token：", token);
            // token
            if (StringUtils.isBlank(token)) {
                var result = HttpResult.unauthorized("非法请求参数，认证失败");
                response.getWriter().write(BeanUtils.toJsonStr(result));
                LogUtil.info("[INFINITY] 非法请求参数，认证失败");
                return false;
            }
            Optional<InfinityAccountRespVO> matchAccount = accounts.stream()
                    .filter(account -> BasicAuthUtil.checkSignature(token, account.getUsername(), account.getPassword())).findFirst();
            if (matchAccount.isEmpty()) {
                var result = HttpResult.result(HttpStatus.HTTP_NOT_AUTHORITATIVE, "用户名密码，认证失败");
                response.getWriter().write(BeanUtils.toJsonStr(result));
                LogUtil.info("[INFINITY] 用户名密码，认证失败");
                return false;
            }
            LogUtil.info("[INFINITY] 认证成功！");
            // 设置当前Infinity账户
            InfinitySession.setCurrentAccount(matchAccount.get());
            if (infinityAuthListener != null) {
                infinityAuthListener.preCompleteHandle(matchAccount.get());
            }
            return true;
        } catch (Exception e) {
            var result = HttpResult.internalServerError("认证异常，请联系管理员");
            response.getWriter().write(BeanUtils.toJsonStr(result));
            LogUtil.error("[INFINITY] 认证异常：", e);
            return false;
        } finally {
            // todo ...
        }
    }

    @Override
    public void postHandle(@Nonnull HttpServletRequest request, @Nonnull HttpServletResponse response, @Nonnull Object handler, ModelAndView modelAndView) {
        if (infinityAuthListener != null) {
            infinityAuthListener.postHandle();
        }
        InfinitySession.clearCurrentAccount();
    }

    @Override
    public void afterCompletion(@Nonnull HttpServletRequest request, @Nonnull HttpServletResponse response, @Nonnull Object handler, Exception e) {
    }
}
