package com.elitesland.cbpl.infinity.web.config;

import com.elitesland.cbpl.infinity.web.http.service.InfinityRestClient;
import com.elitesland.cbpl.infinity.web.security.service.oauth2.OAuth2Client;
import com.elitesland.cbpl.tool.core.http.RestWrapper;
import com.elitesland.cbpl.tool.redis.util.RedisUtil;
import io.netty.channel.ChannelOption;
import io.netty.handler.timeout.ReadTimeoutHandler;
import io.netty.handler.timeout.WriteTimeoutHandler;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.http.client.reactive.ReactorClientHttpConnector;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.reactive.function.client.ExchangeStrategies;
import org.springframework.web.reactive.function.client.WebClient;
import reactor.netty.http.client.HttpClient;

import java.time.Duration;
import java.util.concurrent.TimeUnit;

import static com.elitesland.cbpl.infinity.web.config.InfinityWebProperties.*;

/**
 * @author eric.hao
 * @since 2021/07/13
 */
@Slf4j
@Configuration
@EnableConfigurationProperties(InfinityWebProperties.class)
@ConditionalOnProperty(prefix = INFINITY_WEB_CONFIG_PREFIX, name = "enabled", havingValue = "true")
@RequiredArgsConstructor
public class InfinityWebAutoConfiguration {

    @Bean
    @ConditionalOnMissingBean
    public WebClient webClient(InfinityWebProperties properties) {
        HttpClient httpClient = HttpClient.create()
                // 连接超时
                .option(ChannelOption.CONNECT_TIMEOUT_MILLIS, CONNECT_TIMEOUT)
                // 响应超时
                .responseTimeout(Duration.ofMillis(RESPONSE_TIMEOUT))
                .doOnConnected(conn -> conn
                        // 读写超时
                        .addHandlerLast(new ReadTimeoutHandler(READ_TIMEOUT, TimeUnit.MILLISECONDS))
                        .addHandlerLast(new WriteTimeoutHandler(WRITE_TIMEOUT, TimeUnit.MILLISECONDS))
                );
        return WebClient.builder()
                .clientConnector(new ReactorClientHttpConnector(httpClient))
                .exchangeStrategies(ExchangeStrategies.builder()
                        .codecs(clientCodecConfigurer -> clientCodecConfigurer
                                .defaultCodecs()
                                // 设置缓冲区大小，单位，字节(B) => MB 转 B
                                .maxInMemorySize(properties.getMaxInMemorySize() * 1024 * 1024))
                        .build())
                .build();
    }

    @Bean
    public InfinityRestClient infinityRestClient(WebClient webClient) {
        return new InfinityRestClient(webClient);
    }

    @Bean
    @ConditionalOnClass
    public OAuth2Client oAuth2Client(RestWrapper restWrapper, RedisUtil redisUtil) {
        return new OAuth2Client(restWrapper, redisUtil);
    }

    @Bean("phoenixRestTemplate")
    @ConditionalOnMissingBean(name = "phoenixRestTemplate")
    public RestTemplate phoenixRestTemplate() {
        HttpComponentsClientHttpRequestFactory httpRequestFactory = new HttpComponentsClientHttpRequestFactory();
        httpRequestFactory.setConnectionRequestTimeout(RESPONSE_TIMEOUT);
        httpRequestFactory.setConnectTimeout(CONNECT_TIMEOUT);
        httpRequestFactory.setReadTimeout(READ_TIMEOUT);
        return new RestTemplate(httpRequestFactory);
    }
}
