package com.elitesland.cbpl.infinity.web.config;

import com.elitesland.cbpl.infinity.web.config.provider.DnsCacheHandler;
import com.elitesland.cbpl.infinity.web.http.service.InfinityRestClient;
import com.elitesland.cbpl.infinity.web.security.service.oauth2.OAuth2Client;
import com.elitesland.cbpl.tool.core.http.RestWrapper;
import com.elitesland.cbpl.tool.redis.util.RedisUtil;
import io.netty.channel.*;
import io.netty.handler.timeout.ReadTimeoutHandler;
import io.netty.handler.timeout.WriteTimeoutHandler;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.http.client.reactive.ReactorClientHttpConnector;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.reactive.function.client.ExchangeStrategies;
import org.springframework.web.reactive.function.client.WebClient;
import reactor.netty.http.client.HttpClient;
import reactor.netty.resources.ConnectionProvider;

import java.time.Duration;
import java.util.concurrent.TimeUnit;

import static com.elitesland.cbpl.infinity.web.config.InfinityWebProperties.*;

/**
 * @author eric.hao
 * @since 2021/07/13
 */
@Slf4j
@Configuration
@EnableConfigurationProperties(InfinityWebProperties.class)
@ConditionalOnProperty(prefix = INFINITY_WEB_CONFIG_PREFIX, name = "enabled", havingValue = "true")
@RequiredArgsConstructor
public class InfinityWebAutoConfiguration {

    /**
     * 连接池配置
     *
     * @since 0.3.6-SNAPSHOT
     */
    @Bean
    public ConnectionProvider connectionProvider(InfinityWebProperties properties) {
        return ConnectionProvider.builder(properties.getConnectionName())
                .maxConnections(properties.getMaxConnections())
                .pendingAcquireTimeout(Duration.ofSeconds(properties.getPendingAcquireTimeout()))
                .build();
    }

    @Bean
    @ConditionalOnMissingBean
    public WebClient webClient(InfinityWebProperties properties, ConnectionProvider connectionProvider) {
        HttpClient httpClient = HttpClient.create(connectionProvider)
                // 连接超时
                .option(ChannelOption.CONNECT_TIMEOUT_MILLIS, CONNECT_TIMEOUT)
                // 响应超时
                .responseTimeout(Duration.ofMillis(RESPONSE_TIMEOUT))
                .doOnConnected(conn -> {
                    // 新增DNS配置，单位秒
                    if (properties.isDnsCacheEnabled()) {
                        conn.addHandlerFirst(new DnsCacheHandler(properties.getDnsTtl(), properties.getDnsNegativeTtl()));
                    }
                    // 读写超时
                    conn.addHandlerLast(new ReadTimeoutHandler(READ_TIMEOUT, TimeUnit.MILLISECONDS));
                    conn.addHandlerLast(new WriteTimeoutHandler(WRITE_TIMEOUT, TimeUnit.MILLISECONDS));
                })
                // 增加流量监控（需配合日志级别DEBUG）
                .wiretap(properties.isWiretap())
                // 新增连接保活配置（保持长连接减少握手次数，需配合连接池使用）
                .keepAlive(properties.isKeepAlive());
        return WebClient.builder()
                .clientConnector(new ReactorClientHttpConnector(httpClient))
                .exchangeStrategies(ExchangeStrategies.builder()
                        .codecs(clientCodecConfigurer -> clientCodecConfigurer
                                .defaultCodecs()
                                // 设置缓冲区大小，单位，字节(B) => MB 转 B
                                .maxInMemorySize(properties.getMaxInMemorySize() * 1024 * 1024))
                        .build())
                .build();
    }

    @Bean
    public InfinityRestClient infinityRestClient(WebClient webClient) {
        return new InfinityRestClient(webClient);
    }

    @Bean
    @ConditionalOnClass
    public OAuth2Client oAuth2Client(RestWrapper restWrapper, RedisUtil redisUtil) {
        return new OAuth2Client(restWrapper, redisUtil);
    }

    @Bean("phoenixRestTemplate")
    @ConditionalOnMissingBean(name = "phoenixRestTemplate")
    public RestTemplate phoenixRestTemplate() {
        HttpComponentsClientHttpRequestFactory httpRequestFactory = new HttpComponentsClientHttpRequestFactory();
        httpRequestFactory.setConnectionRequestTimeout(RESPONSE_TIMEOUT);
        httpRequestFactory.setConnectTimeout(CONNECT_TIMEOUT);
        httpRequestFactory.setReadTimeout(READ_TIMEOUT);
        return new RestTemplate(httpRequestFactory);
    }
}
