package com.elitesland.cbpl.infinity.web.config;

import io.swagger.annotations.ApiModelProperty;
import lombok.Data;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.boot.context.properties.ConfigurationProperties;

import static com.elitesland.cbpl.infinity.web.config.InfinityWebProperties.INFINITY_WEB_CONFIG_PREFIX;

/**
 * @author eric.hao
 * @since 2023/04/16
 */
@Data
@ConfigurationProperties(prefix = INFINITY_WEB_CONFIG_PREFIX)
public class InfinityWebProperties implements InitializingBean {

    public static final String INFINITY_WEB_CONFIG_PREFIX = "cbpl.infinity.web";

    /**
     * 是否启用：默认关闭
     */
    private boolean enabled;

    /**
     * 超时值为0，指无限时
     */
    @ApiModelProperty("响应超时（以毫秒为单位）")
    private int responseTimeout = 3000;

    /**
     * 超时值为0，指无限时
     */
    @ApiModelProperty("连接超时（以毫秒为单位）")
    private int connectTimeout = 3000;

    /**
     * 超时值为0，指无限时
     */
    @ApiModelProperty("套接字读取超时（以毫秒为单位）")
    private int readTimeout = 5000;

    /**
     * 超时值为0，指无限时
     */
    @ApiModelProperty("套接字写入超时（以毫秒为单位）")
    private int writeTimeout = 5000;

    @ApiModelProperty("重试机制")
    private RetryProperty retry = new RetryProperty();

    /**
     * 缓冲区默认1M，WebClient默认是256K
     */
    @ApiModelProperty("缓冲区大小（单位MB）")
    private int maxInMemorySize = 1;

    /**
     * 连接池名称
     *
     * @since 0.3.6-SNAPSHOT
     */
    @ApiModelProperty("连接池名称")
    private String connectionName = "WebClientTp";

    /**
     * 同时可以有多少个活跃的连接
     *
     * @since 0.3.6-SNAPSHOT
     */
    @ApiModelProperty("最大连接数")
    private int maxConnections = 200;

    /**
     * 当连接池耗尽时，请求等待获取连接的最长时间，单位秒<br/>
     * 超过这个时间就会报错：IllegalStateException: Timeout waiting for connection
     *
     * @since 0.3.6-SNAPSHOT
     */
    @ApiModelProperty("等待获取超时")
    private int pendingAcquireTimeout = 45;

    /**
     * 开启DNS解析缓存配置：dnsTtl、dnsNegativeTtl 才会生效 <br/>
     * 开启后，会将DNS解析结果缓存起来，缓存时间由 dnsTtl 和 dnsNegativeTtl 决定 <br/>
     * 关闭后，每次请求都会重新解析DNS <br/>
     *
     * Since: 0.3.6-SNAPSHOT
     */
    @ApiModelProperty("是否开启DNS解析缓存：默认关闭")
    private boolean dnsCacheEnabled;

    /**
     * DNS解析缓存: 有效缓存时间，单位秒
     *
     * @since 0.3.6-SNAPSHOT
     */
    @ApiModelProperty("有效缓存时间")
    private int dnsTtl = 600;

    /**
     * DNS解析缓存: 失败结果的有效缓存时间，单位秒
     *
     * @since 0.3.6-SNAPSHOT
     */
    @ApiModelProperty("失败结果的有效缓存时间")
    private int dnsNegativeTtl = 30;

    /**
     * 需配合日志级别DEBUG，防止日志文件过大
     *
     * @since 0.3.6-SNAPSHOT
     */
    @ApiModelProperty("是否开启流量监控日志，默认关闭")
    private boolean wiretap;

    /**
     * 保持长连接减少握手次数，需配合连接池使用
     *
     * @since 0.3.6-SNAPSHOT
     */
    @ApiModelProperty("是否保持长连接")
    private boolean keepAlive = true;

    @Data
    public static class RetryProperty {
        @ApiModelProperty("失败重试次数")
        private int retryTimes = 0;
    }

    public static boolean INFINITY_WEB_ENABLED;
    public static int RESPONSE_TIMEOUT;
    public static int CONNECT_TIMEOUT;
    public static int READ_TIMEOUT;
    public static int WRITE_TIMEOUT;
    public static int RETRY_TIMES;

    @Override
    public void afterPropertiesSet() throws Exception {
        INFINITY_WEB_ENABLED = enabled;
        RESPONSE_TIMEOUT = responseTimeout;
        CONNECT_TIMEOUT = connectTimeout;
        READ_TIMEOUT = readTimeout;
        WRITE_TIMEOUT = writeTimeout;
        RETRY_TIMES = retry.getRetryTimes();
    }
}
