package com.elitesland.cbpl.infinity.web.security.service.weaver.e9.service;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.crypto.asymmetric.RSA;
import com.elitesland.cbpl.infinity.server.security.service.InfinitySecurityService;
import com.elitesland.cbpl.infinity.server.security.vo.param.InfinitySecuritySaveParamVO;
import com.elitesland.cbpl.infinity.web.security.service.weaver.convert.WeaverAccountConvert;
import com.elitesland.cbpl.infinity.web.security.service.weaver.domain.WeaverAccountVO;
import com.elitesland.cbpl.infinity.web.security.service.weaver.domain.WeaverRegisterVO;
import com.elitesland.cbpl.infinity.web.security.service.weaver.domain.WeaverTokenVO;
import com.elitesland.cbpl.infinity.web.security.service.weaver.e9.util.WeaverE9Key;
import com.elitesland.cbpl.infinity.web.security.service.weaver.e9.util.WeaverE9Util;
import com.elitesland.cbpl.logging.syslog.util.LogUtil;
import com.elitesland.cbpl.tool.core.bean.BeanUtils;
import com.elitesland.cbpl.tool.redis.util.RedisUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.web.reactive.function.client.WebClient;

import java.util.concurrent.TimeUnit;

import static com.elitesland.cbpl.infinity.web.security.service.weaver.constant.WeaverApiConstant.WEAVER_E9_AUTH_GET_TOKEN;
import static com.elitesland.cbpl.infinity.web.security.service.weaver.constant.WeaverApiConstant.WEAVER_E9_AUTH_REGISTER;

/**
 * @author eric.hao
 * @since 2024/08/22
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class WeaverE9ServiceImpl implements WeaverE9Service {

    private final WebClient webClient;
    private final RedisUtil redisUtil;
    private final InfinitySecurityService securityService;

    @Override
    public void register(WeaverAccountVO accountVO) {
        var saveParam = WeaverAccountConvert.INSTANCE.voToSaveParam(accountVO);
        var detail = securityService.securityQueryOne(saveParam);
        // 1. 未持久化，重新注册
        if (ObjectUtil.isNull(detail)) {
            createRSA(saveParam, accountVO);
            return;
        }

        // 2. 读配置，或重新获取
        String publicKey = detail.getLocalPublicKey();
        String privateKey = detail.getLocalPrivateKey();
        String spk = detail.getServerPublicKey();
        String secret = detail.getServerSecret();

        // 重新获取 RSA
        if (StrUtil.isBlank(publicKey) || StrUtil.isBlank(privateKey)) {
            RSA rsa = new RSA();
            publicKey = rsa.getPublicKeyBase64();
            privateKey = rsa.getPrivateKeyBase64();
        }
        // 重新获取 SPK
        if (StrUtil.isBlank(spk) || StrUtil.isBlank(secret)) {
            var result = register(accountVO.getServerUrl(), accountVO.getAppid(), publicKey);
            spk = result.getSpk();
            secret = result.getSecret();
        }
        // 持久化、存储缓存
        saveParam.setId(detail.getId());
        refresh(saveParam, accountVO.getPrefix(), publicKey, privateKey, spk, secret);
    }

    @Override
    public void refreshRegister(WeaverAccountVO accountVO) {
        var saveParam = WeaverAccountConvert.INSTANCE.voToSaveParam(accountVO);
        var detail = securityService.securityQueryOne(saveParam);
        if (ObjectUtil.isNull(detail)) {
            // 报错：未注册过
            LogUtil.info("[BPMN][WEAVER-E9] 未注册许可证，不支持刷新");
            throw new RuntimeException("未注册许可证，不支持刷新");
        }
        saveParam.setId(detail.getId());
        createRSA(saveParam, accountVO);
    }

    @Override
    public String getAccessToken(WeaverAccountVO accountVO) {
        String tokenKey = WeaverE9Util.tokenKey(accountVO.getAppid());
        String cacheToken = redisUtil.getString(tokenKey);
        // 1. 缓存生效
        if (ObjectUtil.isNotNull(cacheToken)) {
            LogUtil.info(tokenKey, "[BPMN][WEAVER-E9] 缓存Token: ", cacheToken);
            return cacheToken;
        }
        // 2. 从缓存获取ECOLOGY系统公钥和Secret
        String spk = WeaverE9Key.getServerPublicKey(accountVO.getPrefix());
        String secret = WeaverE9Key.getServerSecret(accountVO.getPrefix());
        // 如果为空，说明还未进行注册，调用注册接口进行注册认证与数据更新
        if (ObjectUtil.isNull(secret) || ObjectUtil.isNull(spk)) {
            register(accountVO);
            // 重新获取最新ECOLOGY系统公钥和Secret信息
            spk = WeaverE9Key.getServerPublicKey(accountVO.getPrefix());
            secret = WeaverE9Key.getServerSecret(accountVO.getPrefix());
        }
        // 3. 对秘钥进行加密传输，防止篡改数据
        String encryptSecret = WeaverE9Util.encrypt(accountVO.getPrefix(), secret);
        // 4. 获取token
        String resultStr = webClient.post()
                .uri(accountVO.getServerUrl() + WEAVER_E9_AUTH_GET_TOKEN)
                .header("appid", accountVO.getAppid())
                .header("secret", encryptSecret)
                .header("time", accountVO.getTime())
                .exchangeToMono(response -> response.bodyToMono(String.class))
                .block();
        if (StrUtil.isBlank(resultStr)) {
            LogUtil.info(tokenKey, "[BPMN][WEAVER-E9] 获取新Token失败");
            throw new RuntimeException("[WEAVER] 获取新Token失败");
        }
        WeaverTokenVO result = BeanUtils.toBean(resultStr, WeaverTokenVO.class);
        if (result.failed()) {
            LogUtil.info(tokenKey, "[BPMN][WEAVER-E9] 认证异常: ", result.getMsg());
            throw new RuntimeException("Weaver Auth error: " + result.getMsg());
        }
        LogUtil.info(tokenKey, "[BPMN][WEAVER-E9] 获取新Token: ", result.getToken());
        // 5. 写入缓存，有效时长减5分钟
        redisUtil.set(tokenKey, result.getToken(), accountVO.getExpiresIn(), TimeUnit.SECONDS);
        return result.getToken();
    }

    /**
     * 重新生成 RSA
     */
    private void createRSA(InfinitySecuritySaveParamVO saveParam, WeaverAccountVO accountVO) {
        // 重新获取 RSA
        RSA rsa = new RSA();
        String publicKey = rsa.getPublicKeyBase64();
        String privateKey = rsa.getPrivateKeyBase64();
        // 重新获取 SPK
        WeaverRegisterVO registerVO = register(accountVO.getServerUrl(), accountVO.getAppid(), publicKey);
        // 持久化、存储缓存
        refresh(saveParam, accountVO.getPrefix(), publicKey, privateKey, registerVO.getSpk(), registerVO.getSecret());
    }

    /**
     * 调用ECOLOGY系统接口进行注册
     */
    private WeaverRegisterVO register(String serverUrl, String appId, String publicKey) {
        String resultStr = webClient.post()
                .uri(serverUrl + WEAVER_E9_AUTH_REGISTER)
                .header("appid", appId)
                .header("cpk", publicKey)
                .exchangeToMono(response -> response.bodyToMono(String.class))
                .block();
        if (StrUtil.isBlank(resultStr)) {
            throw new RuntimeException("[WEAVER] 注册许可失败");
        }
        return BeanUtils.toBean(resultStr, WeaverRegisterVO.class);
    }

    /**
     * 持久化、更新缓存
     */
    private void refresh(InfinitySecuritySaveParamVO saveParam, String prefix, String publicKey, String privateKey, String spk, String secret) {
        saveParam.setLocalPublicKey(publicKey);
        saveParam.setLocalPrivateKey(privateKey);
        saveParam.setServerPublicKey(spk);
        saveParam.setServerSecret(secret);
        securityService.save(saveParam);
        WeaverE9Key.setCache(prefix, spk, secret);
    }
}
