package com.elitesland.cbpl.logging.common.pipeline.mysql.service;

import cn.hutool.core.date.LocalDateTimeUtil;
import com.elitesland.cbpl.logging.audit.data.vo.param.AuditQueryParamVO;
import com.elitesland.cbpl.logging.audit.domain.AuditLogVO;
import com.elitesland.cbpl.logging.common.config.LoggingProperty;
import com.elitesland.cbpl.logging.common.data.vo.param.TrackDocQueryParamVO;
import com.elitesland.cbpl.logging.common.data.vo.resp.TrackDocRespVO;
import com.elitesland.cbpl.logging.common.domain.TrackDoc;
import com.elitesland.cbpl.logging.common.pipeline.PipelineService;
import com.elitesland.cbpl.logging.common.pipeline.mysql.entity.TrackDocDO;
import com.elitesland.cbpl.logging.common.pipeline.mysql.repo.MysqlTrackDocRepo;
import com.elitesland.cbpl.logging.common.pipeline.mysql.repo.MysqlTrackDocRepoProc;
import com.elitesland.cbpl.tool.core.bean.BeanUtils;
import com.elitesland.cbpl.tool.core.util.StringUtils;
import com.elitesland.cbpl.tool.db.PagingVO;
import com.elitesland.cbpl.unicom.annotation.UnicomTag;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.List;
import java.util.stream.Collectors;

import static cn.hutool.core.date.DatePattern.NORM_DATETIME_PATTERN;
import static com.elitesland.cbpl.logging.common.domain.TrackType.LOG_AUDIT;

/**
 * @author eric.hao
 * @since 2024/01/30
 */
@Slf4j
@Service
@RequiredArgsConstructor
@UnicomTag("MYSQL")
public class MysqlPipelineService implements PipelineService {

    private final MysqlTrackDocRepo mysqlTrackDocRepo;
    private final MysqlTrackDocRepoProc mysqlTrackDocRepoProc;

    @Transactional(rollbackFor = Exception.class, propagation = Propagation.REQUIRES_NEW)
    @Override
    public boolean consume(LoggingProperty property, TrackDoc trackDoc) {
        try {
            TrackDocDO trackDocDO = saveParamToDO(trackDoc);
            mysqlTrackDocRepo.save(trackDocDO);
            return true;
        } catch (Throwable throwable) {
            logger.error("[PHOENIX-LOG] persistence mysql error", throwable);
        }
        return false;
    }

    @Override
    public PagingVO<TrackDocRespVO> trackDocPageBy(TrackDocQueryParamVO query) {
        long count = mysqlTrackDocRepoProc.trackDocCountBy(query);
        if (count > 0) {
            var list = mysqlTrackDocRepoProc.trackDocPageBy(query);
            return new PagingVO<>(count, list);
        }
        return new PagingVO<>();
    }

    @Override
    public List<TrackDocRespVO> trackDocByParam(TrackDocQueryParamVO query) {
        return mysqlTrackDocRepoProc.trackDocByParam(query);
    }

    @Override
    public List<AuditLogVO> auditTrackByParam(AuditQueryParamVO query) {
        TrackDocQueryParamVO queryParam = new TrackDocQueryParamVO();
        queryParam.setTrackType(LOG_AUDIT.getCode());
        List<TrackDocRespVO> docs = trackDocByParam(queryParam);
        // TODO 得换个DB-SQL过滤的方法
        return docs.stream().map(doc -> BeanUtils.toBean(doc.getEventParam(), AuditLogVO.class))
                .filter(row -> query.getBizTypes().contains(row.getBizType()))
                .filter(row -> row.getBizId().equals(query.getBizId()))
                .collect(Collectors.toList());
    }

    private TrackDocDO saveParamToDO(TrackDoc saveParam) {
        if (saveParam == null) {
            return null;
        }
        TrackDocDO trackDocDO = new TrackDocDO();
        if (saveParam.getCreateTime() != null) {
            LocalDateTime createTime = LocalDateTimeUtil.parse(saveParam.getCreateTime(), NORM_DATETIME_PATTERN);
            trackDocDO.setCreateTime(createTime);
        }
        trackDocDO.setTraceId(saveParam.getTraceId());
        trackDocDO.setRequestUrl(saveParam.getRequestUrl());
        trackDocDO.setRequestMethod(saveParam.getRequestMethod());
        trackDocDO.setRequestHeader(BeanUtils.toJsonOrEmpty(saveParam.getRequestHeader()));
        trackDocDO.setRequestParams(BeanUtils.toJsonOrEmpty(saveParam.getRequestParams()));
        trackDocDO.setRequestBody(BeanUtils.toJsonOrEmpty(saveParam.getRequestBody()));
        trackDocDO.setResponseData(saveParam.getResponseData());
        trackDocDO.setTrackType(saveParam.getTrackType());
        trackDocDO.setAddressIp(saveParam.getAddressIp());
        trackDocDO.setEventMessage(StringUtils.exceed(saveParam.getEventMessage()));
        trackDocDO.setEventParam(BeanUtils.toJsonOrEmpty(saveParam.getEventParam()));
        trackDocDO.setErrorMessage(saveParam.getErrorMessage());
        return trackDocDO;
    }
}
