package com.elitesland.cbpl.logging.common.service;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.extra.servlet.ServletUtil;
import com.elitesland.cbpl.logging.audit.domain.AuditLogVO;
import com.elitesland.cbpl.logging.audit.spi.AuditLogListener;
import com.elitesland.cbpl.logging.common.config.LoggingProperty;
import com.elitesland.cbpl.logging.common.domain.TrackDoc;
import com.elitesland.cbpl.logging.common.pipeline.PipelineService;
import com.elitesland.cbpl.tool.core.date.DateUtils;
import com.elitesland.cbpl.tool.core.http.HttpServletUtil;
import com.elitesland.cbpl.tool.core.http.RequestWrapper;
import com.elitesland.cbpl.tool.tenant.TenantAsyncSpi;
import com.elitesland.cbpl.unicom.annotation.UnicomTag;
import com.elitesland.cbpl.unicom.domain.InvokeTag;
import com.elitesland.cbpl.unicom.util.UnicomClient;
import com.lzhpo.tracer.util.TracerUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;
import java.util.function.Consumer;

import static com.elitesland.cbpl.tool.core.http.HttpServletUtil.currentRequestWrapper;

/**
 * @author eric.hao
 * @since 2023/08/28
 */
@Slf4j
@Service
@RequiredArgsConstructor
@UnicomTag(InvokeTag.TAG_GENERAL)
public class PersistenceServiceImpl implements PersistenceService {

    @Resource
    private TenantAsyncSpi tenantAsyncSpi;

    @Override
    public <T> void addSystemLogs(LoggingProperty property, String trackType, String eventMessage, List<T> eventParam, String eventError) {
        Consumer<Object> addSystemLog = param -> write(property, currentRequestWrapper(), trackType, eventMessage, param, eventError);
        // 提交日志至线程池
        if (property.isThreadPool()) {
            if (tenantAsyncSpi == null) {
                // TODO 使用其他默认线程方法
                eventParam.forEach(addSystemLog);
            } else {
                tenantAsyncSpi.runAsync(() -> eventParam.forEach(addSystemLog));
            }
        }
        // 主线程执行
        else {
            eventParam.forEach(addSystemLog);
        }
    }

    @Override
    public <T> void addSystemLog(LoggingProperty property, RequestWrapper request, String trackType, String eventMessage, T eventParam, String eventError) {
        // 提交日志至线程池
        if (property.isThreadPool()) {
            if (tenantAsyncSpi == null) {
                // TODO 使用其他默认线程方法
                write(property, request, trackType, eventMessage, eventParam, eventError);
            } else {
                tenantAsyncSpi.runAsync(() -> write(property, request, trackType, eventMessage, eventParam, eventError));
            }
        }
        // 主线程执行
        else {
            write(property, request, trackType, eventMessage, eventParam, eventError);
        }
    }

    @Autowired(required = false)
    private AuditLogListener auditLogListener;

    private final PipelineService pipelineService;

    private void write(LoggingProperty property, RequestWrapper request, String trackType, String eventMessage, Object eventParam, String eventError) {
        if (ObjectUtil.isNull(property.getLogType())) {
            logger.error("[PHOENIX-LOG] EVENT({}) 日志类型未正确配置：{}", trackType, property);
            return;
        }
        if (ObjectUtil.isNull(property.getConsumerPipeline())) {
            logger.error("[PHOENIX-LOG] EVENT({}) 日志消费方式未指定：{}", trackType, property);
            return;
        }
        String logType = property.getLogType().getCode();
        String pipeline = property.getConsumerPipeline().getCode();
        if (!property.isEnabled()) {
            logger.warn("[PHOENIX-LOG] EVENT({}) {} 功能未开启.", trackType, logType);
            return;
        }

        // 构建日志记录内容
        TrackDoc trackDoc = new TrackDoc();
        if (ObjectUtil.isNotNull(request)) {
            trackDoc.setRequestUrl(request.getRequestURL().toString());
            trackDoc.setRequestMethod(request.getMethod());
            trackDoc.setRequestHeader(HttpServletUtil.getHeaders(request));
            trackDoc.setRequestParams(HttpServletUtil.getParameters(request));
            trackDoc.setRequestBody(request.getBodyString());
            trackDoc.setAddressIp(ServletUtil.getClientIP(request));
        }
        trackDoc.setTraceId(TracerUtils.getTraceId());
        trackDoc.setTrackType(trackType);
        trackDoc.setCreateTime(DateUtils.nowStr());
        trackDoc.setEventMessage(eventMessage);
        trackDoc.setEventParam(eventParam);
        trackDoc.setErrorMessage(eventError);

        int maxRetryTimes = property.getRetry().getRetryTimes();
        // 发送日志本地监听
        boolean iOperationLogGetResult = false;
        if (auditLogListener != null && eventParam instanceof AuditLogVO) {
            for (int retryTimes = 0; retryTimes <= maxRetryTimes; retryTimes++) {
                try {
                    iOperationLogGetResult = auditLogListener.createLog((AuditLogVO) eventParam);
                    if (iOperationLogGetResult) {
                        break;
                    }
                } catch (Throwable throwable) {
                    logger.error("[PHOENIX-LOG] send auditLog error", throwable);
                }
            }
            if (!iOperationLogGetResult) {
                auditLogListener.operationLogGetErrorHandler();
            }
        }

        // 发送消息管道
        boolean pipelineServiceResult = false;
        for (int retryTimes = 0; retryTimes <= maxRetryTimes; retryTimes++) {
            try {
                pipelineServiceResult = UnicomClient.supply(() -> pipelineService.consume(property, trackDoc), pipeline);
                if (pipelineServiceResult) {
                    break;
                }
            } catch (Throwable throwable) {
                logger.error("[PHOENIX-LOG] send persistence error", throwable);
            }
        }
        if (!pipelineServiceResult && auditLogListener != null) {
            auditLogListener.pipelineErrorHandler();
        }
    }
}
