package com.elitesland.cbpl.logging.audit.annotation.aspect;

import com.elitesland.cbpl.logging.audit.annotation.AuditLog;
import com.elitesland.cbpl.logging.audit.context.AuditLogContext;
import com.elitesland.cbpl.logging.audit.domain.AuditLogVO;
import com.elitesland.cbpl.logging.common.domain.LogRecord;
import com.elitesland.cbpl.logging.common.service.PersistenceService;
import com.lzhpo.tracer.util.TracerUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.springframework.stereotype.Component;
import org.springframework.util.StopWatch;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import static com.elitesland.cbpl.logging.common.config.LoggingProperties.LOG_AUDIT_CFG;
import static com.elitesland.cbpl.logging.common.constant.TrackType.LOG_AUDIT;

/**
 * @author eric.hao
 * @since 2023/08/29
 */
@Slf4j
@Aspect
@Component
@RequiredArgsConstructor
public class AuditLogAspect {

    private final AuditLogHandler auditLogHandler;
    private final PersistenceService persistenceService;

    @Around("@annotation(com.elitesland.cbpl.logging.audit.annotation.AuditLog) || @annotation(com.elitesland.cbpl.logging.audit.annotation.AuditLogs)")
    public Object around(ProceedingJoinPoint joinPoint) throws Throwable {
        // 原方法执行结果
        Object result;
        // 计算原方法执行时长
        StopWatch stopWatch = null;

        // 审计日志履历
        Map<AuditLog, AuditLogVO> auditLogMap = new LinkedHashMap<>();
        try {
            // 方法执行前日志切面
            auditLogHandler.executeBeforeFunc(joinPoint, auditLogMap);
            stopWatch = new StopWatch();
            stopWatch.start();

            // 原方法执行
            result = joinPoint.proceed();

            stopWatch.stop();
            // 方法成功执行后日志切面
            auditLogHandler.executeAfterFunc(joinPoint, auditLogMap, result);
        }
        // 原方法执行异常
        catch (Throwable throwable) {
            if (stopWatch != null) {
                stopWatch.stop();
            }
            // 方法异常执行后日志切面
            auditLogHandler.executeException(joinPoint, auditLogMap, throwable);
            // 抛出原方法异常
            throw throwable;
        } finally {
            try {
                List<AuditLogVO> auditLogList = new ArrayList<>(auditLogMap.values());
                LogRecord logRecord = new LogRecord();
                logRecord.setTrackType(LOG_AUDIT.getCode());
                // logRecord.setTraceId(TracerUtils.getTraceId());
                logRecord.setMessage("业务审计日志");
                persistenceService.addSystemLogs(LOG_AUDIT_CFG, logRecord, auditLogList);
                // 清除Context：每次方法执行一次
                AuditLogContext.clearContext();
            } catch (Throwable throwableFinal) {
                logger.error("[PHOENIX-AUDIT] around final error", throwableFinal);
            }
        }
        return result;
    }
}
