package com.elitesland.cbpl.logging.audit.annotation.aspect;

import com.elitesland.cbpl.logging.audit.annotation.AuditLog;
import com.elitesland.cbpl.logging.audit.context.AuditLogContext;
import com.elitesland.cbpl.logging.audit.domain.AuditLogVO;
import com.elitesland.cbpl.tool.aop.AopAssistant;
import com.elitesland.cbpl.tool.core.bean.BeanUtils;
import com.elitesland.cbpl.tool.core.map.MapUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.springframework.stereotype.Component;

import java.util.Arrays;
import java.util.Map;
import java.util.function.Function;

/**
 * @author eric.hao
 * @since 2023/08/30
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class AuditLogHandler {

    private final AuditLogParse auditLogParse;

    /**
     * 方法执行前日志切面
     */
    public void executeBeforeFunc(ProceedingJoinPoint joinPoint, Map<AuditLog, AuditLogVO> auditLogMap) {
        try {
            AuditLog[] annotations = AopAssistant.getAnnotations(joinPoint, AuditLog.class);
            Function<AuditLog, Boolean> executeBeforeFunc = AuditLog::executeBeforeFunc;
            // 将前置和后置执行的注解分开处理并保证最终写入顺序
            Arrays.stream(annotations).filter(executeBeforeFunc::apply).forEach(annotation -> {
                AuditLogVO auditLogVO = auditLogParse.resolve(annotation, joinPoint);
                MapUtils.putIfPresent(auditLogMap, annotation, auditLogVO);
            });
        } catch (Throwable throwableBeforeFunc) {
            logger.error("[PHOENIX-AUDIT] around before function, error:", throwableBeforeFunc);
        }
    }

    /**
     * 方法成功执行后日志切面
     */
    public void executeAfterFunc(ProceedingJoinPoint joinPoint, Map<AuditLog, AuditLogVO> auditLogMap, Object result) {
        try {
            AuditLog[] annotations = AopAssistant.getAnnotations(joinPoint, AuditLog.class);
            Function<AuditLog, Boolean> executeAfterFunc =  (annotation) -> !annotation.executeBeforeFunc();
            // 在LogRecordContext中写入执行后信息
            AuditLogContext.putVariable(AuditLogContext.CONTEXT_KEY_NAME_RETURN, result);
            Arrays.stream(annotations).filter(executeAfterFunc::apply).forEach(annotation -> {
                AuditLogVO auditLogVO = auditLogParse.resolve(annotation, joinPoint);
                MapUtils.putIfPresent(auditLogMap, annotation, auditLogVO);
            });
            // 写入成功执行后日志
            auditLogMap.forEach((annotation, auditLog) -> {
                // 若自定义成功失败，则auditLog.getSuccess非null
                if (auditLog.getSuccess() == null) {
                    auditLog.setSuccess(true);
                }
                if (annotation.recordReturnValue() && result != null) {
                    auditLog.setReturnStr(BeanUtils.toJsonStr(result));
                }
            });
        } catch (Throwable throwableAfterFuncSuccess) {
            logger.error("[PHOENIX-AUDIT] around after function success, error:", throwableAfterFuncSuccess);
        }
    }

    /**
     * 方法异常执行后日志切面
     */
    public void executeException(ProceedingJoinPoint joinPoint, Map<AuditLog, AuditLogVO> auditLogMap, Throwable throwable) {
        try {
            AuditLog[] annotations = AopAssistant.getAnnotations(joinPoint, AuditLog.class);
            Function<AuditLog, Boolean> executeAfterFunc =  (annotation) -> !annotation.executeBeforeFunc();
            // 在LogRecordContext中写入执行后信息
            AuditLogContext.putVariable(AuditLogContext.CONTEXT_KEY_NAME_ERROR_MSG, throwable.getMessage());
            Arrays.stream(annotations).filter(executeAfterFunc::apply).forEach(annotation -> {
                AuditLogVO auditLogVO = auditLogParse.resolve(annotation, joinPoint);
                MapUtils.putIfPresent(auditLogMap, annotation, auditLogVO);
            });
            // 写入异常执行后日志
            auditLogMap.forEach((annotation, auditLog) -> {
                auditLog.setSuccess(false);
                auditLog.setException(throwable.getMessage());
            });
        } catch (Throwable throwableAfterFuncFailure) {
            logger.error("[PHOENIX-AUDIT] around after function failure, error:", throwableAfterFuncFailure);
        }
    }
}
