package com.elitesland.cbpl.logging.infinity.service;

import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.logging.common.domain.LogRecord;
import com.elitesland.cbpl.logging.common.service.PersistenceService;
import com.elitesland.cbpl.logging.infinity.domain.InfinityLogVO;
import com.elitesland.cbpl.logging.infinity.spi.InfinityLogListener;
import com.elitesland.cbpl.tool.core.http.HttpServletUtil;
import com.elitesland.cbpl.tool.core.http.ResponseWrapper;
import com.elitesland.cbpl.tool.tenant.TenantSpiUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.server.ServerHttpRequest;
import org.springframework.http.server.ServerHttpResponse;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.Map;

import static com.elitesland.cbpl.logging.common.config.LoggingProperties.LOG_INFINITY_CFG;
import static com.elitesland.cbpl.logging.common.constant.TrackType.LOG_INFINITY_INVOKE;
import static com.elitesland.cbpl.logging.common.constant.TrackType.LOG_INFINITY_PASSIVE;

/**
 * @author eric.hao
 * @since 2023/06/13
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class InfinityTrackServiceImpl implements InfinityTrackService {

    private final PersistenceService persistenceService;
    @Autowired(required = false)
    private InfinityLogListener infinityLogListener;

    /**
     * 接口(被动)调用日志
     */
    @Override
    public void record(ServerHttpRequest request, ServerHttpResponse response, Object responseBody) {
        ResponseWrapper responseWrapper = HttpServletUtil.wrapper(response);
        Map<String, Object> param = new HashMap<>();
        param.put("CURRENT_TENANT", TenantSpiUtil.currentTenantCode());
        param.put("RESPONSE_STATUS", responseWrapper.getStatus());
        param.put("RESPONSE_BODY", responseBody);

        LogRecord logRecord = new LogRecord();
        logRecord.setTrackType(LOG_INFINITY_PASSIVE.getCode());
        logRecord.setMessage("[INFINITY] 接口(被动)调用日志");
        logRecord.setParam(param);
        persistenceService.addSystemLog(LOG_INFINITY_CFG, HttpServletUtil.wrapper(request), logRecord);
    }

    /**
     * 接口(主动)调用日志
     */
    @Override
    public void record(InfinityLogVO requestParam) {
        String trackType = StrUtil.blankToDefault(requestParam.getBusinessType(), LOG_INFINITY_INVOKE.getCode());
        LogRecord logRecord = new LogRecord();
        logRecord.setBizKey(requestParam.getBusinessKey());
        logRecord.setTrackType(trackType);
        logRecord.setMessage("[INFINITY] 接口(主动)调用日志");
        logRecord.setParam(requestParam);
        persistenceService.addSystemLog(LOG_INFINITY_CFG, logRecord);
        if (infinityLogListener != null) {
            try {
                boolean result = infinityLogListener.createLog(requestParam);
                if (!result) {
                    logger.error("[INFINITY] log listener record failed");
                }
            } catch (Exception e) {
                logger.error("[INFINITY] log listener record error: ", e);
            }
        }
    }
}
