package com.elitesland.cbpl.logging.syslog.util;

import cn.hutool.extra.spring.SpringUtil;
import com.elitesland.cbpl.logging.common.constant.TrackType;
import com.elitesland.cbpl.logging.common.domain.LogRecord;
import com.elitesland.cbpl.logging.common.service.PersistenceService;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import static com.elitesland.cbpl.logging.common.config.LoggingProperties.LOG_SYSLOG_CFG;
import static com.elitesland.cbpl.tool.core.exceptions.ExceptionUtils.formatException;

/**
 * 系统日志 - 工具类
 *
 * @author eric.hao
 * @since 2023/08/17
 */
@Slf4j
@NoArgsConstructor
public class LogUtil {

    private static PersistenceService persistenceService;

    private static PersistenceService getInstance() {
        if (persistenceService == null) {
            persistenceService = SpringUtil.getBean(PersistenceService.class);
        }
        return persistenceService;
    }

    /**
     * 系统日志
     *
     * @param message 执行日志
     */
    public static void info(String message) {
        LogUtil.info(TrackType.LOG_SYSTEM.getCode(), null, message, null, null);
    }

    /**
     * 系统日志
     *
     * @param bizKey  业务主键
     * @param message 执行日志
     */
    public static void info(String bizKey, String message) {
        LogUtil.info(TrackType.LOG_SYSTEM.getCode(), bizKey, message, null, null);
    }

    /**
     * 系统日志
     *
     * @param bizKey  业务主键
     * @param message 执行日志
     * @param param   业务参数
     */
    public static void info(String bizKey, String message, Object param) {
        LogUtil.info(TrackType.LOG_SYSTEM.getCode(), bizKey, message, param, null);
    }

    /**
     * 错误日志
     *
     * @param message 执行日志
     * @param ex      异常错误
     */
    public static void error(String message, Throwable ex) {
        LogUtil.error(TrackType.LOG_SYSTEM.getCode(), null, message, null, ex, null);
    }

    /**
     * 错误日志
     *
     * @param bizKey  业务主键
     * @param message 执行日志
     * @param ex      异常错误
     */
    public static void error(String bizKey, String message, Throwable ex) {
        LogUtil.error(TrackType.LOG_SYSTEM.getCode(), bizKey, message, null, ex, null);
    }

    /**
     * 错误日志
     *
     * @param bizKey  业务主键
     * @param message 执行日志
     * @param param   业务参数
     * @param ex      异常错误
     */
    public static void error(String bizKey, String message, Object param, Throwable ex) {
        LogUtil.error(TrackType.LOG_SYSTEM.getCode(), bizKey, message, param, ex, null);
    }

    /**
     * 系统日志
     *
     * @param trackType 业务类型(自定义传值)
     * @param bizKey    业务主键
     * @param message   执行日志
     * @param param     业务参数
     * @param traceId   指定链路ID
     */
    public static void info(String trackType, String bizKey, String message, Object param, String traceId) {
        addSystemLogs(trackType, bizKey, message, param, null, traceId);
    }

    /**
     * 错误日志
     *
     * @param trackType 业务类型(自定义传值)
     * @param bizKey    业务主键
     * @param message   执行日志
     * @param param     业务参数
     * @param ex        异常错误
     * @param traceId   指定链路ID
     */
    public static void error(String trackType, String bizKey, String message, Object param, Throwable ex, String traceId) {
        addSystemLogs(trackType, bizKey, message, param, formatException(ex), traceId);
    }

    /**
     * 记录日志
     */
    private static void addSystemLogs(String trackType, String bizKey, String message, Object eventParam, String trackTypeError, String traceId) {
        LogRecord logRecord = new LogRecord();
        logRecord.setBizKey(bizKey);
        logRecord.setTrackType(trackType);
        // logRecord.setTraceId(traceId);
        logRecord.setMessage(message);
        logRecord.setParam(eventParam);
        logRecord.setErrorMessage(trackTypeError);
        getInstance().addSystemLog(LOG_SYSLOG_CFG, logRecord);
    }
}
