package com.elitesland.cbpl.logging.common.service;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.extra.servlet.ServletUtil;
import com.elitescloud.boot.context.TenantSession;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import com.elitesland.cbpl.cloudt.tenant.TenantWrapper;
import com.elitesland.cbpl.logging.audit.domain.AuditLogVO;
import com.elitesland.cbpl.logging.audit.spi.AuditLogListener;
import com.elitesland.cbpl.logging.common.config.LoggingProperty;
import com.elitesland.cbpl.logging.common.domain.TrackDoc;
import com.elitesland.cbpl.tool.core.date.DateUtils;
import com.elitesland.cbpl.tool.core.http.HttpServletUtil;
import com.elitesland.cbpl.tool.core.http.RequestWrapper;
import com.elitesland.cbpl.tool.es.repository.ElasticsearchRepository;
import com.elitesland.cbpl.unicom.annotation.UnicomTag;
import com.elitesland.cbpl.unicom.domain.InvokeTag;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.skywalking.apm.toolkit.trace.TraceContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;

import static com.elitesland.cbpl.tool.core.http.HttpServletUtil.currentRequestWrapper;
import static com.elitesland.cbpl.tool.es.util.ElasticsearchUtil.indexNameByDay;

/**
 * @author eric.hao
 * @since 2023/08/28
 */
@Slf4j
@Service
@RequiredArgsConstructor
@UnicomTag(InvokeTag.TAG_GENERAL)
public class PersistenceServiceImpl implements PersistenceService {

    private static final String LOG_TEMPLATE = "[PHOENIX-LOG] {} 暂不支持{}存储.";

    private final TenantWrapper tenantWrapper;
    @Resource
    private ElasticsearchRepository elasticRepository;

    @Override
    public <T> void addSystemLogs(LoggingProperty property, String event, List<T> params, String eventError) {
        Consumer<Object> addSystemLog = param -> write(property, currentRequestWrapper(), event, param, eventError);
        // 提交日志至线程池
        if (property.isThreadPool()) {
            tenantWrapper.runAsync(() -> params.forEach(addSystemLog));
        }
        // 主线程执行
        else {
            params.forEach(addSystemLog);
        }
    }

    @Override
    public void addSystemLog(LoggingProperty property, String event, Object param, String eventError) {
        addSystemLog(property, currentRequestWrapper(), event, param, eventError);
    }

    @Override
    public void addSystemLog(LoggingProperty property, RequestWrapper request, String event, Object param, String eventError) {
        // 提交日志至线程池
        if (property.isThreadPool()) {
            tenantWrapper.runAsync(() -> write(property, request, event, param, eventError));
        }
        // 主线程执行
        else {
            write(property, request, event, param, eventError);
        }
    }

    @Autowired(required = false)
    private AuditLogListener auditLogListener;

    private void write(LoggingProperty property, RequestWrapper request, String event, Object param, String eventError) {
        if (ObjectUtil.isNull(property.getLogType())) {
            logger.error("[PHOENIX-LOG] EVENT({}) 日志类型未正确配置：{}", event, property);
            return;
        }
        if (ObjectUtil.isNull(property.getConsumerPipeline())) {
            logger.error("[PHOENIX-LOG] EVENT({}) 日志消费方式未指定：{}", event, property);
            return;
        }
        String logType = property.getLogType().getCode();
        String pipeline = property.getConsumerPipeline().getCode();
        if (!property.isEnabled()) {
            logger.info("[PHOENIX-LOG] EVENT({}) {} 功能未开启.", event, logType);
            return;
        }

        // 构建日志记录内容
        TrackDoc trackDoc = new TrackDoc();
        if (ObjectUtil.isNotNull(request)) {
            trackDoc.setRequestUrl(request.getRequestURL());
            trackDoc.setRequestMethod(request.getMethod());
            trackDoc.setRequestHeader(HttpServletUtil.getHeaders(request));
            trackDoc.setRequestParams(HttpServletUtil.getParameters(request));
            trackDoc.setRequestBody(request.getBodyString());
            trackDoc.setAddressIp(ServletUtil.getClientIP(request));
        }
        trackDoc.setTraceId(TraceContext.traceId());
        trackDoc.setTrackType(event);
        trackDoc.setCreateTime(DateUtils.nowStr());
        trackDoc.setEventParam(param);
        trackDoc.setErrorMessage(eventError);

        // pipelineService.execUnicomTag("xx").notify((AuditLogVO) param);
        int maxRetryTimes = property.getRetry().getRetryTimes();
        // 发送日志本地监听
        boolean iOperationLogGetResult = false;
        if (auditLogListener != null) {
            for (int retryTimes = 0; retryTimes <= maxRetryTimes; retryTimes++) {
                try {
                    if (param instanceof AuditLogVO) {
                        iOperationLogGetResult = auditLogListener.createLog((AuditLogVO) param);
                    }
                    if (iOperationLogGetResult) {
                        break;
                    }
                } catch (Throwable throwable) {
                    logger.error("[PHOENIX-AUDIT] send auditLog error", throwable);
                }
            }
            if (!iOperationLogGetResult) {
                auditLogListener.operationLogGetErrorHandler();
            }
        }

        // 发送消息管道
        boolean pipelineServiceResult = false;
        for (int retryTimes = 0; retryTimes <= maxRetryTimes; retryTimes++) {
            try {
                // 通过消息管道配置，进行持久化存储
                switch (property.getConsumerPipeline()) {
//                    case MYSQL:
//                        logger.info(LOG_TEMPLATE, logType, pipeline);
//                        break;
//                    case ORACLE:
//                        logger.info(LOG_TEMPLATE, logType, pipeline);
//                        break;
                    case ES:
                        try {
                            // ES索引必填
                            Assert.notBlank(property.getEsIndexName(), "【ES】日志索引(es-index-name)未配置");
                            // 根据租户情况，按日期规则创建索引
                            String indexName = getIndexName(property.getEsIndexName());
                            // ES生命周期策略
                            String policy = StrUtil.blankToDefault(property.getEsPolicyName(), "");
                            if (elasticRepository.createIndex(indexName, policy, property.getEsIndexName())) {
                                elasticRepository.insert(trackDoc, indexName);
                            }
                            pipelineServiceResult = true;
                        } catch (Throwable throwable) {
                            logger.error("[PHOENIX-AUDIT] persistence es error", throwable);
                        }
                        break;
//                    case MONGO:
//                        logger.info(LOG_TEMPLATE, logType, pipeline);
//                        break;
//                    case ROCKET_MQ:
//                        logger.info(LOG_TEMPLATE, logType, pipeline);
//                        break;
//                    case DISRUPTER:
//                        logger.info(LOG_TEMPLATE, logType, pipeline);
//                        break;
//                    case STREAM:
//                        logger.info(LOG_TEMPLATE, logType, pipeline);
//                        break;
                    default:
                        logger.info(LOG_TEMPLATE, logType, pipeline);
                }
                if (pipelineServiceResult) {
                    break;
                }
            } catch (Throwable throwable) {
                logger.error("[PHOENIX-AUDIT] send auditLog error", throwable);
            }
        }
        if (!pipelineServiceResult && auditLogListener != null) {
            auditLogListener.pipelineErrorHandler();
        }
    }

    /**
     * 根据租户情况，按日期规则创建ES索引
     */
    private String getIndexName(String indexName) {
        Optional<SysTenantDTO> tenantDTO = Optional.ofNullable(TenantSession.getCurrentTenant());
        return tenantDTO.map(tenant -> indexNameByDay(indexName, tenant.getId().toString())).orElseGet(() -> indexNameByDay(indexName));
    }
}
