package com.elitesland.cbpl.logging.audit.annotation.aspect;

import com.elitesland.cbpl.logging.audit.annotation.AuditLog;
import com.elitesland.cbpl.logging.audit.context.AuditLogContext;
import com.elitesland.cbpl.logging.audit.domain.AuditLogVO;
import com.elitesland.cbpl.logging.audit.function.CustomFunctionRegistrar;
import com.elitesland.cbpl.logging.audit.spi.AuditLogListener;
import com.elitesland.cbpl.tool.aop.AopAssistant;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.JoinPoint;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.expression.spel.support.StandardEvaluationContext;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.Map;
import java.util.UUID;

import static com.elitesland.cbpl.logging.audit.util.ExpressionParser.*;

/**
 * @author eric.hao
 * @since 2023/08/29
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class AuditLogParse {

    @Autowired(required = false)
    private AuditLogListener auditLogListener;

    public AuditLogVO resolve(AuditLog annotation, JoinPoint joinPoint) {
        AuditLogVO auditLogVO = null;
        String bizIdSpel = annotation.bizId();
        String bizTypeSpel = annotation.bizType();
        String tagSpel = annotation.tag();
        String msgSpel = annotation.msg();
        String extraSpel = annotation.extra();
        String operatorIdSpel = annotation.operatorId();
        String conditionSpel = annotation.condition();
        String successSpel = annotation.success();
        String bizId = null;
        String bizType = null;
        String tag = null;
        String msg = null;
        String extra = null;
        String operatorId = null;
        Boolean functionExecuteSuccess = null;
        try {
            StandardEvaluationContext context = AuditLogContext.getContext();
            CustomFunctionRegistrar.register(context);
            Map<String, Object> parameters = AopAssistant.getParameters(joinPoint);
            parameters.forEach(context::setVariable);

            // condition 处理：SpEL解析 必须符合表达式
            if (StringUtils.isNotBlank(conditionSpel)) {
                boolean conditionPassed = parseParamToBoolean(conditionSpel, context);
                if (!conditionPassed) {
                    return null;
                }
            }

            // success 处理：SpEL解析 必须符合表达式
            if (StringUtils.isNotBlank(successSpel)) {
                functionExecuteSuccess = parseParamToBoolean(successSpel, context);
            }

            // bizId 处理：SpEL解析 必须符合表达式
            if (StringUtils.isNotBlank(bizIdSpel)) {
                bizId = parseParamToString(bizIdSpel, context);
            }

            // bizType 处理：SpEL解析 必须符合表达式
            if (StringUtils.isNotBlank(bizTypeSpel)) {
                bizType = parseParamToString(bizTypeSpel, context);
            }

            // tag 处理：SpEL解析 必须符合表达式
            if (StringUtils.isNotBlank(tagSpel)) {
                tag = parseParamToString(tagSpel, context);
            }

            // msg 处理：SpEL解析 必须符合表达式 若为实体则JSON序列化实体 若为空则返回null
            if (StringUtils.isNotBlank(msgSpel)) {
                msg = parseParamToStringOrJson(msgSpel, context);
            }

            // extra 处理：SpEL解析 必须符合表达式 若为实体则JSON序列化实体 若为空则返回null
            if (StringUtils.isNotBlank(extraSpel)) {
                extra = parseParamToStringOrJson(extraSpel, context);
            }

            // operatorId 处理：优先级 注解传入 > 自定义接口实现
            if (auditLogListener != null) {
                operatorId = auditLogListener.getOperatorId();
            }
            if (StringUtils.isNotBlank(operatorIdSpel)) {
                operatorId = parseParamToString(operatorIdSpel, context);
            }

            auditLogVO = new AuditLogVO();
            auditLogVO.setLogId(UUID.randomUUID().toString());
            auditLogVO.setBizId(bizId);
            auditLogVO.setBizType(bizType);
            auditLogVO.setTag(tag);
            auditLogVO.setOperateDate(new Date());
            auditLogVO.setMessage(msg);
            auditLogVO.setExtra(extra);
            auditLogVO.setOperatorId(operatorId);
            auditLogVO.setSuccess(functionExecuteSuccess);
            auditLogVO.setAuditDiffVOList(AuditLogContext.getDiffVOList());
        } catch (Exception e) {
            logger.error("[PHOENIX-AUDIT] resolveExpress error", e);
        } finally {
            // 清除Diff实体列表：每次注解执行一次
            AuditLogContext.clearDiffVOList();
        }
        return auditLogVO;
    }
}
