package com.elitesland.cbpl.multilingual.util;

import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.multilingual.db.constant.LangCondition;
import com.querydsl.core.types.*;
import com.querydsl.core.types.dsl.Expressions;
import com.querydsl.core.types.dsl.PathBuilder;
import com.querydsl.core.types.dsl.StringPath;
import com.querydsl.jpa.impl.JPAQuery;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * @author eric.hao
 * @since 2024/12/01
 */
public class MultilingualUtil extends MultilingualCommon {

    /**
     * 对查询字段，进行翻译
     */
    public static void selectTransfer(JPAQuery<?> jpaQuery) {
        FactoryExpression<?> projection = (FactoryExpression<?>) jpaQuery.getMetadata().getProjection();
        List<Expression<?>> args = (projection).getArgs();
        Object[] array = args.stream().map(expression -> {
            // 仅翻译简单字段的查询；如果是复杂表达式(Operation) 则不做处理
            if (!(expression instanceof Path)) {
                return expression;
            }
            Path<?> currPath = (Path<?>) expression;
            PathMetadata metadata = currPath.getMetadata();
            Path<?> rootPath = metadata.getRootPath();
            Class<?> clazz = rootPath.getType(); // DO Class

            if (!translateRequired(clazz, metadata.getName())) {
                return expression;
            }
            PathBuilder<?> entityPath = new PathBuilder(clazz, StrUtil.lowerFirst(clazz.getSimpleName()));
            StringPath path = entityPath.getString("languageTranslation");
//            var expr = MessageFormat.format(
//                    "json_unquote(json_extract({0}, {1}))",
//                    path, translateField(metadata.getName())
//            );
//            return Expressions.stringPath(expr).coalesce((Path<String>) currPath).as(metadata.getName());
            var expr = Expressions.stringTemplate(
                    "json_unquote(json_extract({0}, {1}))",
                    path, translateField(metadata.getName())
            );
            return expr.coalesce((Path<String>) currPath).as(metadata.getName());
        }).toArray();
        Expression<?>[] expressions = Arrays.copyOf(array, array.length, Expression[].class);

        // 重置select字段内容
        jpaQuery.select(Projections.bean(projection.getType(), expressions));
    }

    /**
     * 查询条件，进行equals匹配
     * <li>创建并返回新的查询条件</li>
     */
    public static List<Predicate> predicate(StringPath field, String value) {
        List<Predicate> predicates = new ArrayList<>();
        predicate(predicates, field, value);
        return predicates;
    }

    /**
     * 查询条件，进行equals匹配
     * <li>拼接到已存在的查询条件中</li>
     */
    public static void predicate(List<Predicate> predicates, StringPath field, String value) {
        predicates.add(predicateBuilder(field, value));
    }

    /**
     * 查询条件，进行equals匹配
     * <li>仅拼接单一条件的表达式</li>
     */
    public static Predicate predicateBuilder(StringPath field, String value) {
        return builder(field, value, LangCondition.EQUALS.getCode());
    }

    /**
     * 查询条件，进行like匹配
     * <li>创建并返回新的查询条件</li>
     */
    public static List<Predicate> predicateLike(StringPath field, String value) {
        List<Predicate> predicates = new ArrayList<>();
        predicateLike(predicates, field, value);
        return predicates;
    }

    /**
     * 查询条件，进行like匹配
     * <li>拼接到已存在的查询条件中</li>
     */
    public static void predicateLike(List<Predicate> predicates, StringPath field, String value) {
        predicates.add(predicateLikeBuilder(field, value));
    }

    /**
     * 查询条件，进行like匹配
     * <li>仅拼接单一条件的表达式</li>
     */
    public static Predicate predicateLikeBuilder(StringPath field, String value) {
        return builder(field, value, LangCondition.LIKE.getCode());
    }
}