package com.elitesland.cbpl.multilingual.repo;

import cn.hutool.core.util.ArrayUtil;
import com.elitesland.cbpl.formgenerator.vo.resp.HistoryFieldVO;
import com.elitesland.cbpl.multilingual.common.util.MultilingualUtil;
import com.elitesland.cbpl.multilingual.domain.MultilingualVO;
import com.elitesland.cbpl.tool.core.exceptions.PhoenixException;
import com.querydsl.core.Tuple;
import com.querydsl.core.types.dsl.*;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static cn.hutool.core.text.CharSequenceUtil.toUnderlineCase;
import static com.elitesland.cbpl.multilingual.common.constant.MultilingualConstant.*;

/**
 * @author eric.hao
 * @since 2024/12/03
 */
@Component
@RequiredArgsConstructor
public class MultilingualRepoProc {

    private final JPAQueryFactory jpaQueryFactory;

    /**
     * 通用查询业务数据方法
     *
     * @param tableClass DO类
     * @param bizKey     业务ID
     * @param fields     需要翻译的字段
     * @return 业务数据
     */
    public Map<String, String> query(Class<?> tableClass, Long bizKey, List<HistoryFieldVO> fields) {
        PathBuilder<?> entity = MultilingualUtil.pathBuilder(tableClass);

        // 遍历需要翻译的字段
        StringPath[] paths = new StringPath[]{};
        paths = ArrayUtil.append(paths, entity.getString(DB_FIELD_LANGUAGE));
        for (HistoryFieldVO field : fields) {
            paths = ArrayUtil.append(paths, entity.getString(field.getFieldCode()));
        }

        // 查询业务数据
        NumberPath<Long> id = entity.getNumber(DB_FIELD_ID, Long.class);
        var jpaQuery = jpaQueryFactory.select(paths).from(entity).where(id.eq(bizKey));
        Tuple tuple = jpaQuery.fetchOne();
        if (tuple == null) {
            throw PhoenixException.unexpected("数据(" + bizKey + ")不存在");
        }

        // 转换成Map返回
        Map<String, String> result = new HashMap<>();
        for (StringPath path : paths) {
            result.put(path.getMetadata().getName(), tuple.get(path));
        }
        return result;
    }

    /**
     * 通用更新-单条数据的翻译内容
     *
     * @param tableClass     DO类
     * @param multilingualVO 更新参数
     * @return 有效更新记录条数
     */
    public long update(Class<?> tableClass, MultilingualVO multilingualVO) {
        PathBuilder<?> entity = MultilingualUtil.pathBuilder(tableClass);

        JPAUpdateClause update = jpaQueryFactory.update(entity);
        // 构建参数
        Map<String, List<String>> params = updateParamBuilder(multilingualVO);
        // 更新翻译字段
        StringPath languageTranslation = entity.getString(DB_FIELD_LANGUAGE);
        MultilingualUtil.updateClause(update, languageTranslation, params);
        // 更新条件
        NumberPath<Long> id = entity.getNumber(DB_FIELD_ID, Long.class);
        update.where(id.eq(multilingualVO.getBizKey()));
        // 有效更新记录
        return update.execute();
    }

    /**
     * 按语种进行分组，重新构建 翻译内容的存储结构
     */
    private Map<String, List<String>> updateParamBuilder(MultilingualVO multilingualVO) {
        // key语种编码，value[filed, value, filed, value ... ]
        Map<String, List<String>> fieldParams = new HashMap<>();
        // 按语种进行分组，重新构建 翻译内容的存储结构
        for (var field : multilingualVO.getFields()) {
            for (Map.Entry<String, String> entry : field.getTranslate().entrySet()) {
                List<String> fieldValues = fieldParams.get(entry.getKey());
                // 新数组
                if (fieldValues == null) {
                    fieldValues = new ArrayList<>();
                }
                // 字段名
                fieldValues.add(toUnderlineCase(field.getFieldCode()));
                // 翻译值
                fieldValues.add(entry.getValue());
                fieldParams.put(entry.getKey(), fieldValues);
            }
        }
        return fieldParams;
    }
}
