package com.elitesland.cbpl.multilingual.service;

import cn.hutool.core.collection.CollUtil;
import com.elitesland.cbpl.formgenerator.entity.EnhanceFormDO;
import com.elitesland.cbpl.formgenerator.repo.EnhanceFormRepoProc;
import com.elitesland.cbpl.formgenerator.vo.resp.HistoryFieldVO;
import com.elitesland.cbpl.multilingual.cache.MultilingualCache;
import com.elitesland.cbpl.multilingual.domain.MultilingualFieldVO;
import com.elitesland.cbpl.multilingual.domain.MultilingualVO;
import com.elitesland.cbpl.multilingual.repo.MultilingualRepoProc;
import com.elitesland.cbpl.tool.core.bean.BeanUtils;
import com.elitesland.cbpl.tool.core.exceptions.PhoenixException;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import static cn.hutool.core.text.CharSequenceUtil.toUnderlineCase;
import static com.elitesland.cbpl.multilingual.common.constant.MultilingualConstant.DB_FIELD_LANGUAGE;

/**
 * @author eric.hao
 * @since 2024/12/02
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class MultilingualServiceImpl implements MultilingualService {

    private final MultilingualRepoProc multilingualRepoProc;
    private final EnhanceFormRepoProc enhanceFormRepoProc;

    @SneakyThrows
    @Override
    public MultilingualVO queryTranslate(String formCode, Long bizKey) {
        // 翻译数据
        var multilingualVO = new MultilingualVO();
        multilingualVO.setFormCode(formCode);
        multilingualVO.setBizKey(bizKey);

        // 1.表单配置信息
        EnhanceFormDO formDO = enhanceFormRepoProc.findByFormCode(formCode);
        if (formDO == null) {
            throw PhoenixException.unexpected("表单(" + formCode + ")不存在或未启用");
        }

        // 2.需要翻译的字段
        Class<?> tableClass = Class.forName(formDO.getTableClass());
        List<HistoryFieldVO> fields = MultilingualCache.languageFields(tableClass);

        // 3.查询业务数据
        Map<String, String> data = multilingualRepoProc.query(tableClass, bizKey, fields);

        // 4.1 如果没有翻译内容，直接返回
        String translate = data.get(DB_FIELD_LANGUAGE);
        if (translate == null) {
            return multilingualVO;
        }
        // 4.2 转义翻译内容
        var langPackVO = BeanUtils.toMap(translate, new TypeReference<Map<String, Map<String, String>>>() {
        });

        List<MultilingualFieldVO> fieldList = new ArrayList<>();
        for (HistoryFieldVO field : fields) {
            MultilingualFieldVO fieldVO = new MultilingualFieldVO(); // TODO 默认设置语言列表
            fieldVO.setFieldCode(field.getFieldCode());
            fieldVO.setFieldName(field.getFieldName());
            fieldVO.setFieldValue(data.get(field.getFieldCode()));
            // set multilingual translate
            for (String langKey : langPackVO.keySet()) {
                Map<String, String> fieldValues = langPackVO.get(langKey);
                fieldVO.addTranslation(langKey, fieldValues.get(toUnderlineCase(field.getFieldCode())));
            }
            fieldList.add(fieldVO);
        }
        multilingualVO.setFields(fieldList);
        return multilingualVO;
    }

    @SneakyThrows
    @Transactional(rollbackFor = Exception.class)
    @Override
    public Boolean updateTranslate(MultilingualVO multilingualVO) {
        if (CollUtil.isEmpty(multilingualVO.getFields())) {
            logger.warn("[Multilingual] update multilingual fields is empty]");
            return false;
        }
        // 表单配置信息
        EnhanceFormDO formDO = enhanceFormRepoProc.findByFormCode(multilingualVO.getFormCode());
        if (formDO == null) {
            throw PhoenixException.unexpected("表单(" + multilingualVO.getFormCode() + ")不存在或未启用");
        }
        return multilingualRepoProc.update(Class.forName(formDO.getTableClass()), multilingualVO) > 0;
    }
}
