package com.elitesland.cbpl.multilingual.common.util;

import cn.hutool.core.text.StrPool;
import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.multilingual.cache.MultilingualCache;
import com.elitesland.cbpl.multilingual.common.db.constant.LangCondition;
import com.elitesland.cbpl.multilingual.spi.MultilingualSpi;
import com.elitesland.cbpl.tool.db.SqlUtil;
import com.elitesland.cbpl.tool.extra.spring.SpringUtils;
import com.querydsl.core.types.Path;
import com.querydsl.core.types.PathMetadata;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.dsl.Expressions;
import com.querydsl.core.types.dsl.PathBuilder;
import com.querydsl.core.types.dsl.StringPath;
import lombok.extern.slf4j.Slf4j;

import static cn.hutool.core.text.CharSequenceUtil.toUnderlineCase;
import static com.elitesland.cbpl.multilingual.common.constant.MultilingualConstant.DB_FIELD_LANGUAGE;

/**
 * @author eric.hao
 * @since 2024/12/01
 */
@Slf4j
public abstract class MultilingualCommon {

    /**
     * 当前环境上下文中，需要翻译的语种
     */
    protected static String obtainLanguage() {
        if (SpringUtils.isPresent(MultilingualSpi.class)) {
            var multilingualService = SpringUtils.getBean(MultilingualSpi.class);
            return multilingualService.obtainLanguage();
        }
        logger.warn("[Multilingual] obtainLanguage failed.");
        return null;
    }

    /**
     * 是否为翻译字段
     *
     * @param tableClass 数据库表
     * @param tableField 数据库字段
     */
    protected static boolean translateRequired(Class<?> tableClass, String tableField) {
        var fields = MultilingualCache.languageFields(tableClass);
        return fields.stream().anyMatch(field -> field.getFieldCode().equals(tableField));
    }

    /**
     * 转义，需要翻译的字段
     */
    protected static String translateField(String tableField) {
        if (StrUtil.isEmpty(obtainLanguage())) {
            return "$." + toUnderlineCase(tableField);
        }
        return "$." + obtainLanguage() + StrPool.DOT + toUnderlineCase(tableField);
    }

    /**
     * 拼接查询条件
     *
     * @param field     查询字段
     * @param value     查询匹配值
     * @param condition 匹配规则：精准匹配、模糊匹配
     * @return 查询表达式
     */
    protected static Predicate predicateBuilder(StringPath field, String value, int condition) {
        PathMetadata metadata = field.getMetadata();
        Path<?> rootPath = metadata.getRootPath();
        Class<?> tableClass = rootPath.getType(); // DO Class

        // TODO 后续重构策略，匹配三种规则：
        //  1.字段不需要翻译，默认仅匹配数据库表字段
        //  2.需要匹配翻译字段时，增加全局配置：
        //  2.1 仅匹配当前语种的字段
        //  2.2 匹配所有语种的字段

        // 查询字段，需要匹配翻译
        if (translateRequired(tableClass, metadata.getName())) {
            PathBuilder<?> entity = pathBuilder(tableClass);
            StringPath path = entity.getString(DB_FIELD_LANGUAGE);
            var expr = Expressions.stringTemplate(
                    "json_unquote(json_extract({0}, {1}))",
                    path, translateField(metadata.getName())
            );

            // 精准匹配
            if (condition == LangCondition.EQUALS.getCode()) {
                return field.eq(value).or(expr.eq(value));
            }
            // 模糊匹配
            if (condition == LangCondition.LIKE.getCode()) {
                String likeStr = SqlUtil.toSqlLikeString(value);
                return field.like(likeStr).or(expr.like(likeStr));
            }
        }

        // 精准匹配
        if (condition == LangCondition.EQUALS.getCode()) {
            return field.eq(value);
        }
        // 模糊匹配
        if (condition == LangCondition.LIKE.getCode()) {
            String likeStr = SqlUtil.toSqlLikeString(value);
            return field.like(likeStr);
        }
        // 使用错误：其他未定义规则
        return null;
    }

    /**
     * 初始化PathBuilder
     *
     * @param tableClass DO类
     */
    public static PathBuilder<?> pathBuilder(Class<?> tableClass) {
        return new PathBuilder<>(tableClass, StrUtil.lowerFirst(tableClass.getSimpleName()));
    }
}
