package com.elitesland.cbpl.multilingual.repo;

import cn.hutool.core.util.ArrayUtil;
import com.elitesland.cbpl.formgenerator.vo.resp.HistoryFieldVO;
import com.elitesland.cbpl.multilingual.common.repo.MultilingualQueryFactory;
import com.elitesland.cbpl.multilingual.common.util.MultilingualUtil;
import com.elitesland.cbpl.multilingual.domain.MultilingualVO;
import com.elitesland.cbpl.multilingual.spi.MultilingualPredicateSpi;
import com.elitesland.cbpl.tool.core.exceptions.ExceptionUtils;
import com.elitesland.cbpl.tool.core.exceptions.PhoenixException;
import com.elitesland.cbpl.unicom.util.UnicomClient;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Path;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.dsl.*;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;

import static cn.hutool.core.text.CharSequenceUtil.toUnderlineCase;
import static com.elitesland.cbpl.multilingual.common.constant.MultilingualConstant.*;

/**
 * @author eric.hao
 * @since 2024/12/03
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class MultilingualRepoProc {

    private final JPAQueryFactory jpaQueryFactory;
    private final MultilingualQueryFactory multilingualQueryFactory;
    private final MultilingualPredicateSpi predicateSpi;

    /**
     * 通用查询业务数据方法
     *
     * @param tableClass DO类
     * @param bizKey     业务ID
     * @param fields     需要翻译的字段
     * @return 业务数据
     */
    public Map<String, Object> fetchOne(Class<?> tableClass, Long bizKey, List<HistoryFieldVO> fields) {
        try {
            return multilingualQueryFactory.fetchOne(tableClass,
                    (entity) -> {
                        // 遍历需要翻译的字段
                        StringPath[] paths = new StringPath[]{};
                        paths = ArrayUtil.append(paths, entity.getString(DB_FIELD_LANGUAGE));
                        for (HistoryFieldVO field : fields) {
                            paths = ArrayUtil.append(paths, entity.getString(field.getFieldCode()));
                        }
                        return paths;
                    },
                    (entity) -> {
                        NumberPath<Long> id = entity.getNumber(DB_FIELD_ID, Long.class);
                        return id.eq(bizKey);
                    });
        } catch (Throwable e) {
            Throwable error = ExceptionUtils.getCausedBy(e);
            throw PhoenixException.unexpected(error.getMessage());
        }
    }

    public List<Map<String, Object>> fetch(String formCode, Class<?> tableClass, List<HistoryFieldVO> fields) {
        return multilingualQueryFactory.fetch(tableClass,
                (entity) -> {
                    // 遍历需要翻译的字段
                    Path<?>[] paths = new Path[]{};
                    paths = ArrayUtil.append(paths, entity.getNumber(DB_FIELD_ID, Long.class));
                    paths = ArrayUtil.append(paths, entity.getString(DB_FIELD_LANGUAGE));
                    for (HistoryFieldVO field : fields) {
                        paths = ArrayUtil.append(paths, entity.getString(field.getFieldCode()));
                    }
                    return paths;
                },
                (entity) -> {
                    List<Predicate> predicates = new ArrayList<>();
                    NumberPath<Integer> deleteFlag = entity.getNumber("deleteFlag", Integer.class);
                    predicates.add(deleteFlag.eq(0));
                    // 针对不同的表单，可以自定义过滤条件
                    try {
                        UnicomClient.run(() -> predicateSpi.queryBuilder(entity, predicates), formCode);
                    } catch (Exception e) {
                        logger.warn("[Multilingual] custom predicates failed: {}", e.getMessage());
                    }
                    return ExpressionUtils.allOf(predicates);
                });
    }

    /**
     * 通用更新-单条数据的翻译内容
     *
     * @param tableClass     DO类
     * @param multilingualVO 更新参数
     * @return 有效更新记录条数
     */
    public long update(Class<?> tableClass, MultilingualVO multilingualVO) {
        PathBuilder<?> entity = MultilingualUtil.pathBuilder(tableClass);

        JPAUpdateClause update = jpaQueryFactory.update(entity);
        // 更新条件
        NumberPath<Long> id = entity.getNumber(DB_FIELD_ID, Long.class);
        update.where(id.eq(multilingualVO.getBizKey()));
        AtomicLong updateCount = new AtomicLong();
        // 构建参数
        Map<String, List<String>> params = updateParamBuilder(multilingualVO);
        params.forEach((lang, values) -> {
            // 更新翻译字段
            StringPath languageTranslation = entity.getString(DB_FIELD_LANGUAGE);
            MultilingualUtil.updateClause(update, languageTranslation, lang, values);
            // 有效更新记录
            long cnt = update.execute();
            updateCount.set(updateCount.get() + cnt);
        });
        return updateCount.get();
    }

    /**
     * 按语种进行分组，重新构建 翻译内容的存储结构
     */
    private Map<String, List<String>> updateParamBuilder(MultilingualVO multilingualVO) {
        // key语种编码，value[filed, value, filed, value ... ]
        Map<String, List<String>> fieldParams = new HashMap<>();
        // 按语种进行分组，重新构建 翻译内容的存储结构
        for (var field : multilingualVO.getFields()) {
            for (Map.Entry<String, String> entry : field.getTranslate().entrySet()) {
                List<String> fieldValues = fieldParams.get(entry.getKey());
                // 新数组
                if (fieldValues == null) {
                    fieldValues = new ArrayList<>();
                }
                // 字段名
                fieldValues.add(toUnderlineCase(field.getFieldCode()));
                // 翻译值
                fieldValues.add(entry.getValue());
                fieldParams.put(entry.getKey(), fieldValues);
            }
        }
        return fieldParams;
    }
}
