package com.elitesland.cbpl.multilingual.common.repo;

import cn.hutool.core.collection.CollUtil;
import com.elitesland.cbpl.multilingual.common.util.MultilingualUtil;
import com.querydsl.core.Tuple;
import com.querydsl.core.types.Path;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.dsl.PathBuilder;
import com.querydsl.jpa.impl.JPAQueryFactory;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

/**
 * @author eric.hao
 * @since 2024/12/04
 */
@Component
@RequiredArgsConstructor
public class MultilingualQueryFactory {

    private final JPAQueryFactory jpaQueryFactory;

    /**
     * 查询一条数据
     *
     * @param tableClass 实体类
     * @param fields     查询字段
     * @param predicate  查询条件
     * @return 查询结果
     */
    public Map<String, Object> fetchOne(
            Class<?> tableClass, Function<PathBuilder<?>, Path<?>[]> fields, Function<PathBuilder<?>, Predicate> predicate) {
        PathBuilder<?> entity = MultilingualUtil.pathBuilder(tableClass);
        // 查询字段
        Path<?>[] paths = fields.apply(entity);
        // 查询业务数据
        var jpaQuery = jpaQueryFactory.select(paths)
                .from(entity).where(predicate.apply(entity));
        Tuple tuple = jpaQuery.fetchOne();
        if (tuple == null) {
            throw new IllegalArgumentException("[Multilingual] fetch data empty.");
        }
        // 转换成Map返回
        Map<String, Object> result = new HashMap<>();
        for (Path<?> path : paths) {
            result.put(path.getMetadata().getName(), tuple.get(path));
        }
        return result;
    }

    /**
     * 查询多条数据
     *
     * @param tableClass 实体类
     * @param fields     查询字段
     * @param predicate  查询条件
     * @return 查询结果
     */
    public List<Map<String, Object>> fetch(
            Class<?> tableClass, Function<PathBuilder<?>, Path<?>[]> fields, Function<PathBuilder<?>, Predicate> predicate) {
        PathBuilder<?> entity = MultilingualUtil.pathBuilder(tableClass);
        // 查询字段
        Path<?>[] paths = fields.apply(entity);
        // 查询业务数据
        var jpaQuery = jpaQueryFactory.select(paths)
                .from(entity).where(predicate.apply(entity));
        List<Tuple> tupleList = jpaQuery.fetch();
        // 没查到数据
        if (CollUtil.isEmpty(tupleList)) {
            return new ArrayList<>();
        }
        // 转换成Map返回
        List<Map<String, Object>> result = new ArrayList<>();
        for (Tuple tuple : tupleList) {
            Map<String, Object> rowData = new HashMap<>();
            for (Path<?> path : paths) {
                rowData.put(path.getMetadata().getName(), tuple.get(path));
            }
            result.add(rowData);
        }
        return result;
    }
}
