package com.elitesland.cbpl.multilingual.cache;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitesland.cbpl.formgenerator.service.EnhanceFormFieldService;
import com.elitesland.cbpl.formgenerator.vo.resp.HistoryFieldVO;
import com.elitesland.cbpl.tool.core.bean.BeanUtils;
import com.elitesland.cbpl.tool.core.util.StringUtils;
import com.elitesland.cbpl.tool.extra.spring.SpringUtils;
import com.elitesland.cbpl.tool.redis.util.RedisUtil;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;

import java.time.Duration;
import java.util.Collections;
import java.util.List;

import static cn.hutool.json.JSONUtil.toJsonStr;

/**
 * @author eric.hao
 * @since 2024/12/03
 */
public class MultilingualCache {

    private static final String REDIS_LANGUAGE_KEY = "MULTILINGUAL_CACHE";

    private static final Cache<String, List<HistoryFieldVO>> LANGUAGE_CACHE = Caffeine.newBuilder()
            .maximumSize(2000)
            .expireAfterWrite(Duration.ofMinutes(2))
            .build();

    /**
     * 从缓存中，读取数据库表，需要翻译的字段
     */
    public static List<HistoryFieldVO> languageFields(Class<?> tableClass) {
        String languageKey = StringUtils.tenantKey(tableClass.getName());
        // 1.先从内存中获取
        List<HistoryFieldVO> languageFields = LANGUAGE_CACHE.getIfPresent(languageKey);
        if (CollUtil.isNotEmpty(languageFields)) {
            return languageFields;
        }
        // 2.再从Redis中获取
        List<HistoryFieldVO> data = languageFieldsRedis(tableClass);
        LANGUAGE_CACHE.put(languageKey, data);
        return ObjectUtil.defaultIfNull(data, Collections.emptyList());
    }

    /**
     * 从Redis去读缓存，7天缓存时间
     */
    private static List<HistoryFieldVO> languageFieldsRedis(Class<?> tableClass) {
        String languageKey = StringUtils.tenantKey(tableClass.getName());
        // 从Redis中获取
        RedisUtil redisUtil = SpringUtils.getBean(RedisUtil.class);
        Object result = redisUtil.hget(REDIS_LANGUAGE_KEY, languageKey);
        // 没有数据，读取数据库
        if (result == null) {
            List<HistoryFieldVO> data = languageFieldsDatabase(tableClass);
            if (CollUtil.isEmpty(data)) {
                return Collections.emptyList();
            }
            redisUtil.hset(REDIS_LANGUAGE_KEY, languageKey, data);
            return data;
        }
        return BeanUtils.toList(toJsonStr(result), HistoryFieldVO.class);
    }

    /**
     * 从数据库读取最新数据
     */
    private static List<HistoryFieldVO> languageFieldsDatabase(Class<?> tableClass) {
        String className = tableClass.getName();
        var fieldService = SpringUtils.getBean(EnhanceFormFieldService.class);
        var formVO = fieldService.queryEffectiveByTableClass(className);
        return fieldService.queryFieldsByFormCode(formVO.getFormCode(), formVO.getVersion());
    }

    /**
     * 清除所有缓存
     */
    public static void clearCache() {
        // 清除内存
        LANGUAGE_CACHE.invalidateAll();
        // 清除Redis
        var redisUtil = SpringUtils.getBean(RedisUtil.class);
        redisUtil.del(REDIS_LANGUAGE_KEY);
    }
}
