package com.elitesland.cbpl.multilingual.common.util;

import com.elitesland.cbpl.multilingual.common.db.constant.LangCondition;
import com.elitesland.cbpl.tool.db.SqlUtil;
import com.querydsl.core.types.*;
import com.querydsl.core.types.dsl.*;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAUpdateClause;

import java.util.*;

import static com.elitesland.cbpl.multilingual.common.constant.MultilingualConstant.DB_FIELD_LANGUAGE;

/**
 * @author eric.hao
 * @since 2024/12/01
 */
public class MultilingualUtil extends MultilingualCommon {

    /**
     * 对查询字段，进行翻译
     */
    public static void selectTransfer(JPAQuery<?> jpaQuery) {
        selectTransfer(jpaQuery, null);
    }

    /**
     * 对查询字段，进行翻译
     */
    public static void selectTransfer(JPAQuery<?> jpaQuery, String language) {
        FactoryExpression<?> projection = (FactoryExpression<?>) jpaQuery.getMetadata().getProjection();
        List<Expression<?>> args = (projection).getArgs();
        Object[] array = args.stream().map(expression -> {
            // 仅翻译简单字段的查询；如果是复杂表达式(Operation) 则不做处理
            if (!(expression instanceof Path)) {
                return expression;
            }
            Path<String> currPath = (Path<String>) expression;
            PathMetadata metadata = currPath.getMetadata();
            Path<?> rootPath = metadata.getRootPath();
            Class<?> tableClass = rootPath.getType(); // DO Class

            if (!translateRequired(tableClass, metadata.getName())) {
                return expression;
            }
            PathBuilder<?> entity = MultilingualUtil.pathBuilder(tableClass);
            StringPath path = entity.getString(DB_FIELD_LANGUAGE);
            // 1.提取翻译值：json_extract(language_translation, '$.en.cust_name')
            // 2.去除值中的引号：json_unquote()
            var expr = Expressions.stringTemplate(
                    "cast(json_unquote(json_extract({0}, {1})) as char)",
                    path, translateField(metadata.getName(), language)
            );
            // 3.转义字符串null值：case when `expr` = 'null' then null else `expr` end
            // 4.判断翻译值是否为null：coalesce(`expr`, fieldValue)
            return expr
//                    .when(Expressions.stringTemplate("'null'"))
//                    .then(Expressions.stringTemplate("null"))
//                    .otherwise(expr)
                    // .coalesce(currPath)
                    .as(metadata.getName());
            // 5.完整表达式，示例：
            // coalesce(
            //  case when
            //      json_unquote(json_extract(language_translation, '$.en.cust_name')) = 'null'
            //      then null
            //      else json_unquote(json_extract(language_translation, '$.en.cust_name'))
            //  end
            //  , cust_name
            // ) as cust_name
        }).toArray();
        Expression<?>[] expressions = Arrays.copyOf(array, array.length, Expression[].class);

        // 重置select字段内容
        jpaQuery.select(Projections.bean(projection.getType(), expressions));
    }

    /**
     * 查询条件，进行equals匹配
     * <li>创建并返回新的查询条件</li>
     */
    public static List<Predicate> predicate(StringPath field, String value) {
        List<Predicate> predicates = new ArrayList<>();
        predicate(predicates, field, value);
        return predicates;
    }

    /**
     * 查询条件，进行equals匹配
     * <li>拼接到已存在的查询条件中</li>
     */
    public static void predicate(List<Predicate> predicates, StringPath field, String value) {
        predicates.add(predicateBuilder(field, value));
    }

    /**
     * 查询条件，进行equals匹配
     * <li>仅拼接单一条件的表达式</li>
     */
    public static Predicate predicateBuilder(StringPath field, String value) {
        return predicateBuilder(field, value, LangCondition.EQUALS.getCode());
    }

    /**
     * 查询条件，进行like匹配
     * <li>创建并返回新的查询条件</li>
     */
    public static List<Predicate> predicateLike(StringPath field, String value) {
        List<Predicate> predicates = new ArrayList<>();
        predicateLike(predicates, field, value);
        return predicates;
    }

    /**
     * 查询条件，进行like匹配
     * <li>拼接到已存在的查询条件中</li>
     */
    public static void predicateLike(List<Predicate> predicates, StringPath field, String value) {
        predicates.add(predicateLikeBuilder(field, value));
    }

    /**
     * 查询条件，进行like匹配
     * <li>仅拼接单一条件的表达式</li>
     */
    public static Predicate predicateLikeBuilder(StringPath field, String value) {
        return predicateBuilder(field, value, LangCondition.LIKE.getCode());
    }

    /**
     * 按语种进行分组，对翻译内容进行更新
     */
    public static void updateClause(JPAUpdateClause update, StringPath column, Map<String, List<String>> updateParams) {
        // key语种编码，values[filed, value, filed, value ... ]
        updateParams.forEach((lang, values) -> {
            StringTemplate stringTemplate = Expressions.stringTemplate(
                    // 更新JSON字段的表达式
                    "JSON_SET({0}, {1}, {2})",
                    // 第一个参数：多语言存储字段，如果为空 默认赋值为{}
                    column.coalesce("{}"),
                    // 第二个参数：语言编码
                    "$." + lang,
                    // 第三个参数：翻译内容的对象(JSON_OBJECT)
                    Expressions.stringTemplate(SqlUtil.jsonObjectParam(values.size()), values) //
            );
            update.set(column, stringTemplate);
        });
    }

    /**
     * 按语种进行分组，对翻译内容进行更新
     */
    public static void updateClause(JPAUpdateClause update, StringPath column, String lang, List<String> values) {
        // key语种编码，values[filed, value, filed, value ... ]
        StringTemplate stringTemplate = Expressions.stringTemplate(
                // 更新JSON字段的表达式
                "JSON_SET({0}, {1}, {2})",
                // 第一个参数：多语言存储字段，如果为空 默认赋值为{}
                column.coalesce("{}"),
                // 第二个参数：语言编码
                "$." + lang,
                // 第三个参数：翻译内容的对象(JSON_OBJECT)
                Expressions.stringTemplate(SqlUtil.jsonObjectParam(values.size()), values) //
        );
        update.set(column, stringTemplate);
    }
}